/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

package org.jboss.internal.soa.esb.couriers;

import java.net.URI;
import java.net.URISyntaxException;
import java.util.UUID;

import org.jboss.soa.esb.addressing.Call;
import org.jboss.soa.esb.addressing.EPR;
import org.jboss.soa.esb.addressing.MalformedEPRException;
import org.jboss.soa.esb.addressing.eprs.FileEpr;
import org.jboss.soa.esb.addressing.eprs.JDBCEpr;
import org.jboss.soa.esb.addressing.eprs.JMSEpr;
import org.jboss.soa.esb.couriers.CourierException;
import org.jboss.soa.esb.couriers.CourierFactory;
import org.jboss.soa.esb.couriers.CourierTimeoutException;
import org.jboss.soa.esb.couriers.CourierUtil;
import org.jboss.soa.esb.couriers.TwoWayCourier;
import org.jboss.soa.esb.filter.FilterManager;
import org.jboss.soa.esb.message.Message;

/**
 * A two-way-courier can perform message deliveries and pickups.
 * 
 * @author esteban
 * @author kstam@redhat.com
 */
public class TwoWayCourierImpl implements TwoWayCourier
{
	private DeliverOnlyCourier _deliverCourier;
        
        private EPR _toEPR ;

	private PickUpOnlyCourier _pickupCourier;

	/**
	 * Constructor.
	 * 
	 * @param toEpr -
	 *            to address
	 * @param replyToEpr -
	 *            reply to address
	 * @throws CourierException
	 */
	public TwoWayCourierImpl(EPR toEpr, EPR replyToEpr)
			throws CourierException, MalformedEPRException
	{
		setToEpr(toEpr);
		setReplyToEpr(replyToEpr);
	}

	/**
	 * @see org.jboss.soa.esb.couriers.TwoWayCourier#setToEpr(toEPR).
	 */
	public void setToEpr(EPR toEPR) throws CourierException,
			MalformedEPRException
	{
		DeliverOnlyCourier old = _deliverCourier;
                try
                {
                    _deliverCourier = getDeliverCourier(toEPR);
                    _toEPR = toEPR ;
                }
                finally
                {
                    CourierUtil.cleanCourier(old) ;
                }
	}

	/**
	 * @see org.jboss.soa.esb.couriers.TwoWayCourier#setReplyEpr(toReplyEPR).
	 */
	public void setReplyToEpr(EPR replyToEPR) throws CourierException,
			MalformedEPRException
	{
		PickUpOnlyCourier old = _pickupCourier;
                try
                {
                    _pickupCourier = getPickupCourier(replyToEPR);
                }
                finally
                {
                    CourierUtil.cleanCourier(old) ;
                }
	}

	private DeliverOnlyCourier getDeliverCourier(EPR toEPR)
			throws CourierException, MalformedEPRException
	{
		return (null == toEPR) ? null : (DeliverOnlyCourier) courierFromEpr(
				toEPR, false);
	}

	private PickUpOnlyCourier getPickupCourier(EPR replyToEPR)
			throws CourierException, MalformedEPRException
	{
		return (null == replyToEPR) ? null
				: (PickUpOnlyCourier) courierFromEpr(replyToEPR, true);
	}

	private Object courierFromEpr(EPR epr, boolean pickUpOnly)
			throws CourierException, MalformedEPRException
	{
		if (null == epr)
			return null;
		if (epr instanceof JMSEpr)
			return new JmsCourier((JMSEpr) epr, pickUpOnly);
		if (epr instanceof FileEpr)
			return new FileCourier((FileEpr) epr, pickUpOnly);
		if (epr instanceof JDBCEpr)
			return new SqlTableCourier((JDBCEpr) epr, pickUpOnly);

		// TODO the following is necessary because EPR
		// serialization/deserialization loses type

		return courierFromGenericEPR(epr, pickUpOnly);
	}

	private Object courierFromGenericEPR(EPR epr, boolean pickUpOnly)
			throws CourierException, MalformedEPRException
	{
		String addr = null;

		addr = epr.getAddr().getAddress();
		if (addr.startsWith(JMSEpr.JMS_PROTOCOL))
			return new JmsCourier(new JMSEpr(epr), pickUpOnly);
		if (addr.startsWith(JDBCEpr.JDBC_PROTOCOL))
			return new SqlTableCourier(new JDBCEpr(epr), pickUpOnly);
		// TODO magic strings
		if (addr.startsWith("file://") || addr.startsWith("ftp://")
				|| addr.startsWith("sftp://") || addr.startsWith("ftps://"))
			return new FileCourier(new FileEpr(epr), pickUpOnly);

		throw new MalformedEPRException("Courier for "
				+ epr.getClass().getSimpleName() + " not supported: ESB-unaware EPR used!");
	}

	/**
	 * @see org.jboss.soa.esb.couriers.Courier#deliver(Message message).
	 */
	public boolean deliver(Message message) throws CourierException,
			MalformedEPRException
	{
		if (null == _deliverCourier)
			throw new CourierException("No deliverAsync courier");
                final Call call = message.getHeader().getCall() ;
                call.setTo(_toEPR) ;
                
                final boolean setMessageID = call.getMessageID() == null;
                if (setMessageID)
                {
                    final String messageID = UUID.randomUUID().toString() ;
                    try
                    {
                        call.setMessageID(new URI(messageID)) ;
                    }
                    catch (final URISyntaxException urise)
                    {
                        throw new MalformedEPRException("Failed to set message ID to " + messageID) ;
                    }
                }

                try
                {
                    message = FilterManager.getInstance().doOutputWork(message, null);
                    
                    return _deliverCourier.deliver(message);
                }
                finally
                {
                    if (setMessageID)
                    {
                        call.setMessageID(null) ;
                    }
                }
	}

	/**
	 * @see org.jboss.soa.esb.couriers.TwoWayCourier#pickup(long waitTime).
	 */
	public Message pickup(long waitTime) throws CourierException,
			CourierTimeoutException
	{
		return pickup(waitTime, _pickupCourier);
	}

	/**
	 * @see org.jboss.soa.esb.couriers.TwoWayCourier#pickup(long waitTime, EPR
	 *      epr).
	 */
	public Message pickup(long waitTime, EPR epr) throws CourierException,
			CourierTimeoutException, MalformedEPRException
	{
		return pickup(waitTime, getPickupCourier(epr));
	}

	private Message pickup(long waitTime, PickUpOnlyCourier courier)
			throws CourierException, CourierTimeoutException
	{
		if (null == courier)
			throw new CourierException("No courier defined for pick ups");
		final Message result = courier.pickup(waitTime);
                
                return (result == null ? null : FilterManager.getInstance().doInputWork(result, null)) ;
	}
        
        public void cleanup ()
        {
            CourierUtil.cleanCourier(_deliverCourier) ;
            _deliverCourier = null ;
            CourierUtil.cleanCourier(_pickupCourier) ;
            _pickupCourier = null ;
            CourierFactory.deregisterCourier(this) ;
        }
}
