/* Resource constrained software pipeliner.  This file contains
   interface of the resource constrained software pipeliner.

   Copyright (C) 2003 Free Software Foundation, Inc.

   Written by Vladimir Makarov <vmakarov@redhat.com>
   
   This file is part of GCC.

   GCC is free software; you can redistribute it and/or modify it under
   the terms of the GNU General Public License as published by the Free
   Software Foundation; either version 2, or (at your option) any later
   version.

   GCC is distributed in the hope that it will be useful, but WITHOUT ANY
   WARRANTY; without even the implied warranty of MERCHANTABILITY or
   FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
   for more details.

   You should have received a copy of the GNU General Public License
   along with GCC; see the file COPYING.  If not, write to the Free
   Software Foundation, 59 Temple Place - Suite 330, Boston, MA
   02111-1307, USA.  */

/* RCSP is implemented as an abstract data in file `rcsp-core.c'.  All
   communication of the abstract data and GCC is going through this
   file.  In other words, the file implements all interface of
   `rcsp-core.c' to the compiler.  To read more about RCSP, please see
   comments in `rcsp-core.c'.  */

#include "config.h"
#include "system.h"
#include "coretypes.h"
#include "tm.h"
#include "toplev.h"
#include "rtl.h"
#include "tm_p.h"
#include "hard-reg-set.h"
#include "basic-block.h"
#include "regs.h"
#include "function.h"
#include "flags.h"
#include "insn-config.h"
#include "insn-attr.h"
#include "except.h"
#include "recog.h"
#include "sched-int.h"
#include "target.h"
#include "rcsp-int.h"
#include "hashtab.h"

#if RCSP_SOFTWARE_PIPELINING
#if 0
#define RCSP_DO_NOT_USE_OBSTACK
#endif

#ifndef RCSP_DO_NOT_USE_OBSTACK
#include "obstack.h"
#endif

extern char * reg_known_equiv_p;
extern rtx *  reg_known_value;

/* This page contains functions and data needed for the interface of
   the software pipeliner.  */

/* The following array contains starts of chains of locations
   referenced by the same insn.  */
static struct loc **insn_locs;

/* The following variable is length of the previous array.  */
static int insn_locs_length;

/* This structure is an element of the list which contains references
   for software pipeliner locations.  */
struct loc_link
{
  struct loc *loc;
  struct loc_link *next;
};

/* This structure is reference for a software pipeliner location.  */
struct loc
{
  /* Rtx representation of the location.  */
  rtx loc;
  /* Flag of that the location is used.  */
  int usage_p;
  /* Type of writing if the location is written.  */
  int write_kind;
  /* All the structures which correspond to hard registers or memory are
     chained by the following members.  */
  struct loc *next_hard_reg;
  struct loc *next_mem;
  /* All location references which are in the same insn are chained by
     the following member.  */
  struct loc *next_insn_loc;
  /* Links to locations which correspondingly contain the location and
     are contained in the location.  */
  struct loc_link *supers;
  struct loc_link *subs;
};

/* This structure is a software pipeliner insn.  */
struct insn
{
  /* The order number (0, 1, ...) of software pipeliner insn.  */
  int number;
  /* Source file and line of the insn.  The value of `file' is NULL if
     source file and line is not known.  This is always NULL for dummy
     insn.  */
  char *file;
  int line;
  /* Original insn.  Corresponding insns from different iterations
     have the same member value.  The value is NULL only for dummy
     insn.  */
  rtx orig_insn;
  /* Output rtx representation of the insn.  It means that the
     corresponding insns from different iterations may be different
     (e.g. register in insn may be renamed).  The value is NULL only
     for dummy insn.  */
  rtx insn;
  /* Notes (and labels) which should be output before and after the
     insn.  */
  rtx notes_before;
  rtx notes_after;
  /* The following member value is code label of exit label of the
     loop. The value is not NULL only for stop packets which
     correspond exit jumps from the loop.  */
  rtx exit_label;
  /* The following member value is TRUE if the insn sets up a register
     living after the loop.  */
  int live_reg_set_p;
};

/* The following variable contains array of all software pipeliner
   insns.  Index corresponds to order numbers of insns.  */
static varray_type insns;

/* The following variables contain arrays of locations which are read,
   may be written, are written, and are write barriers.  */
static varray_type insn_reads;
static varray_type insn_possible_writes;
static varray_type insn_write_barriers;
static varray_type insn_writes;

/* The following variable values are actual lengths of the
   corresponding arrays.  */
static int insns_length;
static int insn_reads_length;
static int insn_possible_writes_length;
static int insn_write_barriers_length;
static int insn_writes_length;

/* The following variables refer for all created locations which are a
   hard register and a memory correspondingly.  */
static struct loc *hard_reg_loc_chain, *mem_loc_chain;

/* The following variable refers for location corresponding to all
   memory.  */
static struct loc *all_memory_loc;

/* Forward declarations for mutually recursive functions.  */
static void process_writes (rtx x, insn_ptr insn);


/* The following function adds pointer EL to varray of pointers VECTOR
   which already contains *LENGTH elements.  Then *LENGTH is
   incremented.  */
static void
add_vector_element (varray_type * vector, void * el, int * length)
{
  unsigned size = VARRAY_SIZE (*vector);

  if ((unsigned) *length >= size)
    {
      size *= 2;
      if (size < (unsigned) *length)
	size = (unsigned) *length;
      VARRAY_GROW (*vector, size);
    }
  VARRAY_GENERIC_PTR (*vector, *length) = el;
  (*length)++;
}

/* The following function creates and returns new location link
   structure with given member values LOC and NEXT.  */

static struct loc_link *
new_loc_link (struct loc * loc, struct loc_link * next)
{
  struct loc_link *result = rcsp_malloc (sizeof (* result));

  result->loc = loc;
  result->next = next;
  return result;
}

/* The following function creates and returns new location structure
   with given member values LOC, USAGE_P, WRITE_KIND, and
   NEXT_INSN_LOC.  */

static struct loc *
new_loc (rtx loc, int usage_p, int write_kind, struct loc * next_insn_loc)
{
  struct loc *result = rcsp_malloc (sizeof (struct loc));

  result->loc = loc;
  result->usage_p = usage_p;
  result->write_kind = write_kind;
  result->next_insn_loc = next_insn_loc;
  result->supers = result->subs = NULL;
  return result;
}

/* The following function creates and returns new location structure
   with given member values LOC, USAGE_P, and WRITE_KIND.  The new
   location is added to the list locations of INSN and lists of hard
   register and memory locations if it is applicable to LOC.  */

static struct loc *
new_insn_loc (insn_ptr insn, rtx loc, int usage_p, int write_kind)
{
  struct loc *result;

  result = new_loc (loc, usage_p, write_kind, insn_locs [insn->number]);
  insn_locs [insn->number] = result;
  if (GET_CODE (loc) == REG && REGNO (loc) < FIRST_PSEUDO_REGISTER)
    {
      result->next_hard_reg = hard_reg_loc_chain;
      hard_reg_loc_chain = result;
    }
  else if (GET_CODE (loc) == MEM)
    {
      result->next_mem = mem_loc_chain;
      mem_loc_chain = result;
    }
  return result;
}

/* The following function frees location LOC and loc links of the
   locations.  */

static void
free_loc (struct loc * loc)
{
  struct loc_link *curr_link, *next_link; 

  for (curr_link = loc->supers; curr_link != NULL; curr_link = next_link)
    {
      next_link = curr_link->next;
      rcsp_free (curr_link);
    }
  for (curr_link = loc->subs; curr_link != NULL; curr_link = next_link)
    {
      next_link = curr_link->next;
      rcsp_free (curr_link);
    }
  rcsp_free (loc);
}

/* The following function frees chain of locations starting with
   LOC.  */

static void
free_insn_locs (struct loc * loc)
{
  struct loc *next_loc;

  while (loc != NULL)
    {
      next_loc = loc->next_insn_loc;
      free_loc (loc);
      loc = next_loc;
    }
}

/* The following functions processes the uses of memory and registers
   in rtx X of INSN to set up insn's locations.  */

static void
process_reads (rtx x, insn_ptr insn)
{
  int i;
  int j;
  enum rtx_code code;
  const char *fmt;

  if (x == 0)
    return;

  code = GET_CODE (x);

  switch (code)
    {
    case CONST_INT:
    case CONST_DOUBLE:
    case SYMBOL_REF:
    case CONST:
    case LABEL_REF:
      /* Ignore constants.  Note that we must handle CONST_DOUBLE here
         because it may have a cc0_rtx in its CONST_DOUBLE_CHAIN field, but
         this does not mean that this insn is using cc0.  */
      return;

#ifdef HAVE_cc0
    case CC0:
      /* ??? This code is not implemented because now we do not make
         pipeline for machines with cc0.  */
      new_insn_loc (insn, cc0_rtx, 1 /* usage_p */, 0);
      break;
#endif

    case REG:
      {
	int regno = REGNO (x);

	if (regno < FIRST_PSEUDO_REGISTER)
	  {
	    int i;

	    i = HARD_REGNO_NREGS (regno, GET_MODE (x));
	    while (--i >= 0)
	      new_insn_loc (insn, gen_rtx_REG (VOIDmode, regno + i),
			    1 /* usage_p */, 0);
	  }
	else
	  {
	    new_insn_loc (insn, x, 1 /* usage_p */, 0);

	    /* Pseudos that are REG_EQUIV to something may be replaced
	       by that during reloading.  We need only add dependencies for
	       the address in the REG_EQUIV note.  */
	    if (!reload_completed
		&& reg_known_equiv_p[regno]
		&& GET_CODE (reg_known_value[regno]) == MEM)
	      process_reads (XEXP (reg_known_value[regno], 0), insn);
	  }
	return;
      }

    case MEM:
      if (MEM_VOLATILE_P (x))
	abort ();
      new_insn_loc (insn, x, 1 /* usage_p */, 0);
      /* Take advantage of tail recursion here.  */
      process_reads (XEXP (x, 0), insn);
      return;
      
    /* Make memory barrier to in case a trap handler needs them.  */
    case TRAP_IF:
      new_insn_loc (insn, all_memory_loc->loc,
		    0 /* usage_p */, 2 /* write barrier */);
      break;

    case ASM_OPERANDS:
    case ASM_INPUT:
    case UNSPEC_VOLATILE:
      abort ();

    case PRE_DEC:
    case POST_DEC:
    case PRE_INC:
    case POST_INC:
      /* These both read and modify the result.  We must handle them as writes
         to get proper dependencies for following instructions.  We must handle
         them as reads to get proper dependencies from this to previous
         instructions.  Thus we need to pass them to both process_writes
         and process_reads.  */
      process_reads (XEXP (x, 0), insn);
      process_writes (x, insn);
      return;

    default:
      break;
    }

  /* Other cases: walk the insn.  */
  fmt = GET_RTX_FORMAT (code);
  for (i = GET_RTX_LENGTH (code) - 1; i >= 0; i--)
    {
      if (fmt[i] == 'e')
	process_reads (XEXP (x, i), insn);
      else if (fmt[i] == 'E')
	for (j = 0; j < XVECLEN (x, i); j++)
	  process_reads (XVECEXP (x, i, j), insn);
    }
}

/* The following function processes a single SET, CLOBBER, PRE_DEC,
   POST_DEC, PRE_INC or POST_INC rtx, X, setting up locations of INSN.  */

static void
process_writes (rtx x, insn_ptr insn)
{
  int regno;
  rtx dest = XEXP (x, 0);
  enum rtx_code code = GET_CODE (x);

  if (dest == 0)
    return;

  if (GET_CODE (dest) == PARALLEL
      && GET_MODE (dest) == BLKmode)
    {
      int i;

      for (i = XVECLEN (dest, 0) - 1; i >= 0; i--)
	process_writes (XVECEXP (dest, 0, i), insn);
      if (GET_CODE (x) == SET)
	process_reads (SET_SRC (x), insn);
      return;
    }

  while (GET_CODE (dest) == STRICT_LOW_PART || GET_CODE (dest) == SUBREG
      || GET_CODE (dest) == ZERO_EXTRACT || GET_CODE (dest) == SIGN_EXTRACT)
    {
      if (GET_CODE (dest) == ZERO_EXTRACT || GET_CODE (dest) == SIGN_EXTRACT)
	{
	  /* The second and third arguments are values read by this insn.  */
	  process_reads (XEXP (dest, 1), insn);
	  process_reads (XEXP (dest, 2), insn);
	}
      dest = XEXP (dest, 0);
    }

  if (GET_CODE (dest) == REG)
    {
      int i;

      regno = REGNO (dest);

      /* A hard reg in a wide mode may really be multiple registers.
         If so, mark all of them just like the first.  */
      if (regno < FIRST_PSEUDO_REGISTER)
	{
	  i = HARD_REGNO_NREGS (regno, GET_MODE (dest));
	  while (--i >= 0)
	    {
	      rtx reg = gen_rtx_REG (VOIDmode, regno + i);

	      /* Clobbers need not be ordered with respect to one
		 another, but sets must be ordered with respect to a
		 pending clobber.  */
	      if (code == SET)
		new_insn_loc (insn, reg, 0 /* usage_p */, 0 /* usual write */);
	      else
		new_insn_loc (insn, reg, 0 /* usage_p */, 1 /* may write */);
	    }
	}
      else
	{
	  if (code == SET)
	    new_insn_loc (insn, dest, 0 /* usage_p */, 0 /* usual write */);
	  else
	    new_insn_loc (insn, dest, 0 /* usage_p */, 1 /* may write */);

	  /* Pseudos that are REG_EQUIV to something may be replaced
	     by that during reloading.  We need only add dependencies for
	     the address in the REG_EQUIV note.  */
	  if (!reload_completed
	      && reg_known_equiv_p[regno]
	      && GET_CODE (reg_known_value[regno]) == MEM)
	    process_reads (XEXP (reg_known_value[regno], 0), insn);
	}
    }
  else if (GET_CODE (dest) == MEM)
    {
      /* Writing memory.  */
      if (MEM_VOLATILE_P (dest))
	abort ();
      new_insn_loc (insn, dest, 0 /* usage_p */, 0 /* usual write */);
      process_reads (XEXP (dest, 0), insn);
    }
  
  /* Analyze reads.  */
  if (GET_CODE (x) == SET)
    process_reads (SET_SRC (x), insn);
}

/* The following function processes INSN to set up insn's
   locations.  */

static void
process_insn (insn_ptr insn)
{
  rtx pattern;
  RTX_CODE code;
  int i;

  if (insn == NULL || insn->insn == NULL_RTX)
    /* Dummy op.  */
    abort ();
  pattern = PATTERN (insn->insn);
  code = GET_CODE (insn->insn);
  if (code == CALL
      || (code == NOTE
	  && (NOTE_LINE_NUMBER (insn->insn) == NOTE_INSN_LOOP_BEG
	      || NOTE_LINE_NUMBER (insn->insn) == NOTE_INSN_LOOP_END
	      || NOTE_LINE_NUMBER (insn->insn) == NOTE_INSN_EH_REGION_BEG
	      || NOTE_LINE_NUMBER (insn->insn) == NOTE_INSN_EH_REGION_END
	      /*??? || NOTE_LINE_NUMBER (insn->insn) == NOTE_INSN_SETJMP*/)))
    abort ();

  code = GET_CODE (pattern);
  if (code == COND_EXEC)
    {
      process_reads (COND_EXEC_TEST (pattern), insn);
      pattern = COND_EXEC_CODE (pattern);
      code = GET_CODE (pattern);
    }
  if (code == SET || code == CLOBBER)
    process_writes (pattern, insn);
  else if (code == PARALLEL)
    {
      for (i = XVECLEN (pattern, 0) - 1; i >= 0; i--)
	{
	  rtx sub = XVECEXP (pattern, 0, i);

	  code = GET_CODE (sub);
	  if (code == COND_EXEC)
	    {
	      process_reads (COND_EXEC_TEST (sub), insn);
	      sub = COND_EXEC_CODE (sub);
	      code = GET_CODE (sub);
	    }
	  if (code == SET || code == CLOBBER)
	    process_writes (sub, insn);
	  else
	    process_reads (sub, insn);
	}
    }
  else
    process_reads (pattern, insn);
}

/* The following function creates and returns new insn structure with
   rtx INSN with given attributes source FILE and source LINE.  */

static struct insn *
new_insn (rtx insn, char *file, int line)
{
  struct insn *result = rcsp_malloc (sizeof (struct insn));
  
  result->number = insns_length;
  result->orig_insn = result->insn = insn;
  result->notes_before = result->notes_after = NULL_RTX;
  result->exit_label = NULL_RTX;
  result->live_reg_set_p = FALSE;
  result->file = file;
  result->line = line;
  add_vector_element (&insns, result, &insns_length);
  return result;
}

/* The following variable refers for the table which contains
   information about packets with jumps.  */
static htab_t jumps_table;

/* The structure describes a packet which contains jump.  The labels
   table entry refers for chain of packets which contain jump onto the
   same label.  */
struct jump_packet
{
  /* The following member value is LABEL onto which the jump is made.
     This is the key of jumps_table entry.  */
  rtx label;
  /* The packet which contains the jump.  */
  packet_ptr packet;
  /* The following member value is destination packet after connection
     of the packet with the destination packet.  */
  packet_ptr dest_packet;
  /* The next jump packet with the same label code.  */
  struct jump_packet *next_jump_packet;
  /* The following member is TRUE if we should connect the packet with
     label as jump, otherwise as fall through.  */
  char jump_p;
};

/* The following function returns hash code of JUMP_PACKET.  */
static unsigned
jump_packet_hash (const void *jump_packet)
{
  return (unsigned) ((struct jump_packet *) jump_packet)->label;
}

/* The following function returns TRUE if JUMP_PACKET1 and
   JUMP_PACKET2 refer for the same label.  */

static int
jump_packet_eq (const void *jump_packet1, const void *jump_packet2)
{
  return (((struct jump_packet *) jump_packet1)->label
	  == ((struct jump_packet *) jump_packet2)->label);
}

/* The following function frees jump_packets in chains starting with
   JUMP_PACKET.  */

static void
jump_packet_free (void *jump_packet)
{
  struct jump_packet *curr_jump_packet, *next_jump_packet;

  for (curr_jump_packet = jump_packet;
       curr_jump_packet != NULL;
       curr_jump_packet = next_jump_packet)
    {
      next_jump_packet = curr_jump_packet->next_jump_packet;
      rcsp_free (curr_jump_packet);
    }
}

/* The following function returns information from the label table
   about packets which contains jump onto LABEL.  */

static struct jump_packet *
find_jump_packet (rtx label)
{
  struct jump_packet temp;

  temp.label = label;
  return htab_find (jumps_table, &temp);
}

/* The following function inserts LABEL and PACKET which contains jump
   onto the label into the labels table.  */

static void
insert_jump_packet (rtx label, packet_ptr packet, int jump_p)
{
  void **slot;
  struct jump_packet *jump_packet;

  jump_packet = rcsp_malloc (sizeof (struct jump_packet));
  jump_packet->label = label;
  jump_packet->packet = packet;
  jump_packet->dest_packet = FALSE;
  jump_packet->jump_p = jump_p;
  slot = htab_find_slot (jumps_table, jump_packet, TRUE);
  jump_packet->next_jump_packet = *slot;
  *slot = jump_packet;
}

/* The following function call FUNC for all jump labels table entries
   with additional PARAM.  */

static void
traverse_jumps_table (int (* func) (void **, void *), void * param)
{
  htab_traverse (jumps_table, func, param);
}

/* The following function starts work with the jump labels table.  */

static void
initiate_jumps_table (void)
{
  jumps_table = htab_create (100, jump_packet_hash, jump_packet_eq,
			     jump_packet_free);
}

/* The following function finishes work with the jump labels table.  */

static void
finish_jumps_table (void)
{
  htab_delete (jumps_table);
}

/* The following variable contains registers which live after the
   loop.  */
static bitmap global_regs_living_after_loop;

/* The following function adds global registers living at the start of
   bb containing BB_INSN to set of register living after the loop.  */

static void
update_regs_living_after_loop (rtx bb_insn)
{
  for (;;)
    {
      if (bb_insn == NULL
	  || (GET_CODE (bb_insn) == NOTE
	      && NOTE_LINE_NUMBER (bb_insn) == NOTE_INSN_BASIC_BLOCK))
	break;
      bb_insn = NEXT_INSN (bb_insn);
    }
  if (bb_insn != NULL_RTX)
    {
      basic_block bb = NOTE_BASIC_BLOCK (bb_insn);

      IOR_REG_SET(global_regs_living_after_loop, bb->global_live_at_start);
    }
}

/* The following recursive function returns TRUE if X sets up a register
   living after the loop.  */

static int
rtx_sets_live_reg (rtx x)
{
  int regno, i;
  rtx dest = XEXP (x, 0);

  if (dest == NULL_RTX)
    return FALSE;

  if (GET_CODE (dest) == PARALLEL && GET_MODE (dest) == BLKmode)
    {
      for (i = XVECLEN (dest, 0) - 1; i >= 0; i--)
	if (rtx_sets_live_reg (XVECEXP (dest, 0, i)))
	  return TRUE;
      return FALSE;
    }

  while (GET_CODE (dest) == STRICT_LOW_PART || GET_CODE (dest) == SUBREG
      || GET_CODE (dest) == ZERO_EXTRACT || GET_CODE (dest) == SIGN_EXTRACT)
    dest = XEXP (dest, 0);
  
  if (GET_CODE (dest) == REG)
    {
      regno = REGNO (dest);

      /* A hard reg in a wide mode may really be multiple registers.
         If so, mark all of them just like the first.  */
      if (regno < FIRST_PSEUDO_REGISTER)
	{
	  i = HARD_REGNO_NREGS (regno, GET_MODE (dest));
	  while (--i >= 0)
	    if (REGNO_REG_SET_P (global_regs_living_after_loop, regno + i))
	      return TRUE;
	}
      else if (REGNO_REG_SET_P (global_regs_living_after_loop, regno))
	return TRUE;
    }
  return FALSE;
}

/* The following function returns TRUE if INSN sets up a register
   living after the loop.  */

static int
insn_sets_live_reg_p (rtx insn)
{
  rtx pattern;
  RTX_CODE code;
  int i;

  if (insn == NULL_RTX)
    /* Dummy op.  */
    abort ();
  pattern = PATTERN (insn);
  code = GET_CODE (pattern);
  if (code == SET || code == CLOBBER)
    return rtx_sets_live_reg (pattern);
  else if (code == PARALLEL)
    {
      for (i = XVECLEN (pattern, 0) - 1; i >= 0; i--)
	{
	  code = GET_CODE (XVECEXP (pattern, 0, i));
	  if ((code == SET || code == CLOBBER)
	      && rtx_sets_live_reg (XVECEXP (pattern, 0, i)))
	    return TRUE;
	}
    }
  return FALSE;
}

/* The following function connects packets by backward jumps and for
   loop exits and update live registers after the loop.  START_LABEL
   is label of the loop entry.  */

static int
second_jumps_pass (void ** jump_packet_ptr, void * start_label)
{
  struct jump_packet *curr_jump_packet;
  packet_ptr dest_packet;
  rtx start = start_label;

  dest_packet = NULL;
  /* Find destination packet.  */
  for (curr_jump_packet = *jump_packet_ptr;
       curr_jump_packet != NULL && curr_jump_packet->dest_packet == NULL;
       curr_jump_packet = curr_jump_packet->next_jump_packet)
    ;
  dest_packet
    = (curr_jump_packet == NULL ? NULL : curr_jump_packet->dest_packet);
  /* Make connection by backward jumps and to additional exit
     packets.  */
  for (curr_jump_packet = *jump_packet_ptr;
       curr_jump_packet != NULL;
       curr_jump_packet = curr_jump_packet->next_jump_packet)
    if (curr_jump_packet->dest_packet == NULL
	&& start != curr_jump_packet->label)
      {
	if (dest_packet == NULL)
	  {
	    /* This is an exit packet */
	    dest_packet = create_dummy_packet ();
	    dest_packet->first_op->insn = new_insn (NULL, NULL, 0);
	    dest_packet->stop_packet_p = dest_packet->exit_packet_p = TRUE;
	    dest_packet->first_op->insn->exit_label
	      = curr_jump_packet->label;
	    update_regs_living_after_loop (curr_jump_packet->label);
	  }
	packet_connect (curr_jump_packet->packet, dest_packet, TRUE,
			curr_jump_packet->jump_p);
	curr_jump_packet->dest_packet = dest_packet;
      }
  return TRUE;
}

/* The following function connects PACKET with the previous packet
   *PREV_PACKET (if it is not NULL).  The function also forms chain of
   all packets with the aid of parameter LAST_PACKET.  After that it
   sets up PREV_PACKET and *LAST_PACKET onto PACKET.  */

static void
connect_packets (packet_ptr packet, packet_ptr * prev_packet, packet_ptr * last_packet)
{
  if (*last_packet != NULL)
    (*last_packet)->next_packet = packet;
  if (*prev_packet != NULL)
    {
      if (cond_branch_p ((*prev_packet)->first_op->insn))
	(*prev_packet)->first_op->false_target = packet;
      if (!uncond_branch_p ((*prev_packet)->first_op->insn))
	packet_connect (*prev_packet, packet, FALSE, FALSE);
    }
  *last_packet = *prev_packet = packet;
}

/* The following function connects PACKET with packets in
   LAST_JUMP_PACKET in CFG.  */

static void
connect_jump_packets (packet_ptr packet, struct jump_packet * last_jump_packet)
{
  struct jump_packet *curr_jump_packet;

  for (curr_jump_packet = last_jump_packet;
       curr_jump_packet != NULL;
       curr_jump_packet = curr_jump_packet->next_jump_packet)
    {
      packet_connect (curr_jump_packet->packet, packet, TRUE,
		      curr_jump_packet->jump_p);
      curr_jump_packet->dest_packet = packet;
    }
}

/* The following function redirects packets in chain LAST_JUMP_PACKET
   onto LABEL.  */

static void
redirect_jump_packets (struct jump_packet * last_jump_packet, rtx label)
{
  struct jump_packet *curr_jump_packet;

  for (curr_jump_packet = last_jump_packet;
       curr_jump_packet != NULL;
       curr_jump_packet = curr_jump_packet->next_jump_packet)
    {
      insert_jump_packet (label, curr_jump_packet->packet,
			  curr_jump_packet->jump_p);
      /* Don't process the label on the second jump pass.  */
      curr_jump_packet->dest_packet = curr_jump_packet->packet;
    }
}

/* The following recursive function clears exit packet flag of PACKET
   and all its predecessors.  */

static void
clear_exit_packet_flag (packet_ptr packet)
{
  packet_ptr pred;
  int pred_number;

  if (packet->visit_number1 == last_packet_visit_number1)
    return;
  packet->visit_number1 = last_packet_visit_number1;
  packet->exit_packet_p = FALSE;
  for (pred_number = 0;
       (pred = packet->pred [pred_number]) != NULL;
       pred_number++)
    clear_exit_packet_flag (pred);
}

static void
remove_reg_br_prob (rtx insn)
{
  rtx prev_note, note;

  for (prev_note = NULL_RTX, note = REG_NOTES (insn);
       note;
       prev_note = note, note = XEXP (note, 1))
    if (REG_NOTE_KIND (note) == REG_BR_PROB)
      {
	if (prev_note == NULL_RTX)
	  REG_NOTES (insn) = XEXP (note, 1);
	else
	  XEXP (prev_note, 1) = XEXP (note, 1);
      }
}

/* The following function returns TRUE if INSN contains unconditional
   jump which can be in software pipelined loop.  */

static int
uncond_jump_p (rtx insn)
{
  rtx x = PATTERN (insn);

  if (GET_CODE (x) == PARALLEL)
    x = XVECEXP (x, 0, 0);

  if (GET_CODE (x) != SET)
    return FALSE;
  if (GET_CODE (SET_DEST (x)) != PC)
    return FALSE;
  if (GET_CODE (SET_SRC (x)) == LABEL_REF)
    return TRUE;
  return FALSE;
}

/* The following function returns TRUE if INSN contains conditional
   branch which can be in software pipelined loop.  */

static int
cond_jump_p (rtx insn)
{
  rtx x = PATTERN (insn);

  if (GET_CODE (x) == PARALLEL)
    x = XVECEXP (x, 0, 0);

  if (GET_CODE (x) != SET)
    return FALSE;
  if (GET_CODE (SET_DEST (x)) != PC)
    return FALSE;
  if (GET_CODE (SET_SRC (x)) != IF_THEN_ELSE)
    return FALSE;
  if (XEXP (SET_SRC (x), 2) == pc_rtx
      && (GET_CODE (XEXP (SET_SRC (x), 1)) == LABEL_REF
	  || GET_CODE (XEXP (SET_SRC (x), 1)) == RETURN))
    return TRUE;
  if (XEXP (SET_SRC (x), 1) == pc_rtx
      && (GET_CODE (XEXP (SET_SRC (x), 2)) == LABEL_REF
	  || GET_CODE (XEXP (SET_SRC (x), 2)) == RETURN))
    return TRUE;
  return FALSE;
}

/* The following functions unrolls loop with body [START_INSN,
   BOUND_INSN) RCSP_UNROLL_FACTOR*ITERATION_LOOK_AHEAD + 1 times and
   returns the result as interior packets graph.  Unrolled loop
   contains only copied insn so we can make anything with them without
   disturbing other parts of scheduler.  Notes which should be placed
   at the beginning of pipelined loop are returned through
   START_NOTES.  */

static struct packet *
unroll (rtx start_insn, rtx bound_insn, rtx * start_notes)
{
  rtx insn;
  rtx note;
  RTX_CODE code;
  packet_ptr start_packet;
  packet_ptr stop_packet;
  packet_ptr packet;
  packet_ptr default_prev_packet;
  packet_ptr last_packet;
  struct jump_packet *last_jump_packet;
  varray_type pending_jump_packets;
  int pending_jump_packets_length;
  op_ptr op;
  int i;
  int pred_number;
  packet_ptr copy, pred;
  rtx start_label;
  rtx exit_label;
  rtx notes_before = NULL_RTX;
  rtx notes_after = NULL_RTX;
  rtx insn_copy;
  insn_ptr new = NULL;
  char *file = NULL;
  int line = 0;

  initiate_jumps_table ();
  start_label = prev_nonnote_insn (start_insn);
  if (start_label == NULL_RTX || GET_CODE (start_label) != CODE_LABEL)
    abort ();
  for (exit_label = bound_insn;
       exit_label != NULL_RTX && GET_CODE (exit_label) == BARRIER;
       exit_label = next_nonnote_insn (exit_label))
    ;
  if (exit_label != NULL_RTX && GET_CODE (exit_label) != CODE_LABEL)
    exit_label = NULL_RTX;
  start_packet = NULL;
  stop_packet = create_dummy_packet ();
  stop_packet->first_op->insn = new_insn (NULL, NULL, 0);
  stop_packet->stop_packet_p = stop_packet->exit_packet_p = TRUE;
  stop_packet->main_exit_p = TRUE;
  stop_packet->first_op->insn->exit_label = exit_label;
  default_prev_packet = last_packet = NULL;
  VARRAY_GENERIC_PTR_INIT (pending_jump_packets, 10, "pending jump packets");
  pending_jump_packets_length = 0;
  *start_notes = NULL;
  for (insn = start_insn; insn != NULL_RTX; insn = PREV_INSN (insn))
    if (GET_CODE (insn) == NOTE && NOTE_LINE_NUMBER (insn) > 0)
      {
	line = NOTE_LINE_NUMBER (insn);
	file = (char *) NOTE_SOURCE_FILE (insn);
	break;
      }
  /* First pass: creation of packets and forward connection of them.  */
  for (insn = start_insn; insn != bound_insn; insn = NEXT_INSN (insn))
    {
      code = GET_CODE (insn);
      
      if (code == INSN || (code == JUMP_INSN && !simplejump_p (insn)))
	{
	  new = new_insn (insn, file, line);
	  if (code == JUMP_INSN)
	    {
	      insn_copy = make_jump_insn_raw (copy_insn (PATTERN (new->insn)));
	      JUMP_LABEL (insn_copy) = JUMP_LABEL (new->insn);
	      remove_reg_br_prob (insn_copy);
	    }
	  else
	    insn_copy = make_insn_raw (copy_insn (PATTERN (new->insn)));
	  PREV_INSN (insn_copy) = NEXT_INSN (insn_copy) = NULL_RTX;
	  new->insn = insn_copy;
	  new->notes_before = notes_before;
	  new->notes_after = notes_after;
	  notes_before = notes_after = NULL_RTX;
	  packet = packet_create (-1);
	  op = op_create (new, 0);
	  packet_append_op (packet, op, TRUE);
	  connect_packets (packet, &default_prev_packet, &last_packet);
	  while (pending_jump_packets_length != 0)
	    {
	      pending_jump_packets_length--;
	      connect_jump_packets
		(packet,
		 (struct jump_packet *)
		 VARRAY_GENERIC_PTR (pending_jump_packets,
				     pending_jump_packets_length));
	    }
	  if (start_packet == NULL)
	    start_packet = packet;
	  if (code == JUMP_INSN && !simplejump_p (insn))
	    {
	      op->target_fixed_p = !cond_jump_p (insn);
	      op->can_be_converted_p = !op->target_fixed_p;
	      if (op->can_be_converted_p)
		{
		  /* ??? Now it is switched off.  See comments for
                     `convert_cond_branch'.  */
		  op->can_be_converted_p = FALSE;
#if 0
		  if (!invert_exp (PATTERN (insn), insn))
		    op->can_be_converted_p = FALSE;
		  else if (!invert_exp (PATTERN (insn), insn))
		    /* This should just be putting it back the way it
                       was.  */
		    abort ();
#endif
		}
	      insert_jump_packet (JUMP_LABEL (insn), packet, TRUE);
	      if (uncond_jump_p (insn))
		default_prev_packet = NULL;
	    }
	}
      else if (code == JUMP_INSN)
	{
	  if (!simplejump_p (insn))
	    abort ();
	  if (default_prev_packet == NULL)
	    {
	      /* Jump to Jump.  */
	      if (pending_jump_packets_length == 0)
		abort ();
	      while (pending_jump_packets_length != 0)
		{
		  pending_jump_packets_length--;
		  redirect_jump_packets
		    ((struct jump_packet *)
		     VARRAY_GENERIC_PTR (pending_jump_packets,
					 pending_jump_packets_length),
		     JUMP_LABEL (insn));
		}
	    }
	  else
	    insert_jump_packet (JUMP_LABEL (insn), default_prev_packet, FALSE);
	  default_prev_packet = NULL;
	}
      else if (code == NOTE)
	{
	  if (NOTE_LINE_NUMBER (insn) > 0)
	    {
	      line = NOTE_LINE_NUMBER (insn);
	      file = (char *) NOTE_SOURCE_FILE (insn);
	    }
	  else if (NOTE_LINE_NUMBER (insn) == NOTE_INSN_LOOP_CONT
		   || NOTE_LINE_NUMBER (insn) == NOTE_INSN_LOOP_VTOP)
	    {
	      /* Software pipelined loop are too modified so there is
		 no sense to save such notes.  */
#if 0
	      note = rtx_alloc (NOTE);
	      INSN_UID (note) = INSN_UID (insn);
	      NOTE_SOURCE_FILE (note) = NOTE_SOURCE_FILE (insn);
	      NOTE_LINE_NUMBER (note) = NOTE_LINE_NUMBER (insn);
	      notes_before = alloc_EXPR_LIST (REG_SAVE_NOTE, note,
					      notes_before);
#endif
	    }
	  else if (NOTE_LINE_NUMBER (insn) == NOTE_INSN_DELETED
		   || NOTE_LINE_NUMBER (insn) == NOTE_INSN_DELETED_LABEL
		   || NOTE_LINE_NUMBER (insn) == NOTE_INSN_BLOCK_BEG
		   || NOTE_LINE_NUMBER (insn) == NOTE_INSN_BLOCK_END)
	    {
	      note = rtx_alloc (NOTE);
	      INSN_UID (note) = INSN_UID (insn);
	      NOTE_SOURCE_FILE (note) = NOTE_SOURCE_FILE (insn);
	      NOTE_LINE_NUMBER (note) = NOTE_LINE_NUMBER (insn);
	      CODE_LABEL_NUMBER (note) = CODE_LABEL_NUMBER (insn);
	      *start_notes = alloc_EXPR_LIST (REG_SAVE_NOTE, note,
					      *start_notes);
	    }
	  else if (NOTE_LINE_NUMBER (insn) == NOTE_INSN_LOOP_BEG
		   || NOTE_LINE_NUMBER (insn) == NOTE_INSN_LOOP_END
		   || NOTE_LINE_NUMBER (insn) == NOTE_INSN_EH_REGION_BEG
		   || NOTE_LINE_NUMBER (insn) == NOTE_INSN_EH_REGION_END)
	    abort ();
	}
      else if (code == CODE_LABEL)
	{
	  unsigned size = VARRAY_SIZE (pending_jump_packets);
	  
	  if ((unsigned) pending_jump_packets_length >= size)
	    {
	      size *= 2;
	      if (size < (unsigned) pending_jump_packets_length)
		size = (unsigned) pending_jump_packets_length;
	      VARRAY_GROW (pending_jump_packets, size);
	    }
	  VARRAY_GENERIC_PTR (pending_jump_packets,
			      pending_jump_packets_length)
	    = (void *) find_jump_packet (insn);
	  pending_jump_packets_length++;
	}
      else if (code != BARRIER)
	/* We could nullify DEFAULT_PREV_PACKET here.  But
           unfortunately the barrier for the last loop jump is not in
           the loop part.  */
	abort ();
    }
  global_regs_living_after_loop
    = INIT_REG_SET ((bitmap) rcsp_malloc (sizeof (bitmap_head)));
  CLEAR_REG_SET (global_regs_living_after_loop);
  last_jump_packet = (exit_label == NULL_RTX
		      ? NULL : find_jump_packet (exit_label));
  if (default_prev_packet != NULL || last_jump_packet != NULL)
    update_regs_living_after_loop (bound_insn);
#if 0
  if (default_prev_packet == NULL && last_jump_packet == NULL)
    /* It can not be for infinite loop or loop without main exit.  */
    abort ();
#endif
  connect_packets (stop_packet, &default_prev_packet, &last_packet);
  connect_jump_packets (stop_packet, last_jump_packet);
  /* The 2nd pass: connection of packets by backward jumps and exit
     packets and find registers living after the loop.  */
  traverse_jumps_table (second_jumps_pass, start_label);
  /* Find exit packets: */
  for (packet = start_packet; packet != NULL; packet = packet->next_packet)
    packet->exit_packet_p = TRUE;
  last_packet_visit_number1 = 0;
  for (last_jump_packet = find_jump_packet (start_label);
       last_jump_packet != NULL;
       last_jump_packet = last_jump_packet->next_jump_packet)
    clear_exit_packet_flag (last_jump_packet->packet);
  /* Set up live_reg_set_p for the ops and restore visit numbers.  */
  for (packet = start_packet; packet != NULL; packet = packet->next_packet)
    {
      if (packet->stop_packet_p && !packet->exit_packet_p)
	abort ();
      packet->visit_number1 = -1;
      if (!packet->stop_packet_p)
	packet->first_op->insn->live_reg_set_p
	  = insn_sets_live_reg_p (packet->first_op->insn->insn);
    }
  /* Repeat it RCSP_UNROLL_FACTOR * ITERATION_LOOK_AHEAD times.  */
  for (i = 1; i < RCSP_UNROLL_FACTOR * iteration_look_ahead + 1; i++)
    {
      for (packet = start_packet; packet != NULL; packet = packet->next_packet)
	{
	  if (packet->exit_packet_p)
	    {
	      for (pred_number = 0;
		   (pred = packet->pred [pred_number]) != NULL;
		   pred_number++)
		if (pred->next_frontier_packet != NULL)
		  /* We do not change {true, false} target in possible
		     branch because it is the same after copying.  */
		  packet_connect (pred->next_frontier_packet, packet,
				  FALSE, FALSE);
	    }
	  else
	    {
	      /* Form list of copies: next_frontier_packet is used as
		 the last copy of the packet.  */
	      copy = packet_copy (packet);
	      new = new_insn (packet->first_op->insn->orig_insn,
			      packet->first_op->insn->file,
			      packet->first_op->insn->line);
	      if (packet->first_op->insn->notes_before != NULL_RTX)
		new->notes_before
		  = copy_rtx (packet->first_op->insn->notes_before);
	      if (packet->first_op->insn->notes_after != NULL_RTX)
		new->notes_after
		  = copy_rtx (packet->first_op->insn->notes_after);
	      copy->first_op->insn = new;
	      copy->first_op->insn->live_reg_set_p
		= packet->first_op->insn->live_reg_set_p;
	      if (packet->next_frontier_packet == NULL)
		packet->copy = packet->next_frontier_packet = copy;
	      else
		{
		  packet->next_frontier_packet->copy = copy;
		  packet->next_frontier_packet = copy;
		}
	      copy->min_it_n = i;
	      copy->first_op->it_num = i;
	      insn_copy = packet->first_op->insn->orig_insn;
	      if (GET_CODE (insn_copy) == JUMP_INSN)
		{
		  insn_copy = make_jump_insn_raw (copy_insn (PATTERN
							     (insn_copy)));
		  JUMP_LABEL (insn_copy)
		    = JUMP_LABEL (packet->first_op->insn->orig_insn);
		  remove_reg_br_prob (insn_copy);
		}
	      else
		insn_copy = make_insn_raw (copy_insn (PATTERN (insn_copy)));
	      PREV_INSN (insn_copy) = NEXT_INSN (insn_copy) = NULL_RTX;
	      copy->first_op->insn->insn = insn_copy;
	      for (pred_number = 0;
		   (pred = packet->pred [pred_number]) != NULL;
		   pred_number++)
		{
		  packet_change_dest (pred->next_frontier_packet,
				      packet, packet->next_frontier_packet);
		  packet_change_src (packet->next_frontier_packet, pred,
				     pred->next_frontier_packet);
		}
	    }
	}
    }
  /* Connect iterations.  */
  for (packet = start_packet, i = 1;
       i < RCSP_UNROLL_FACTOR * iteration_look_ahead + 1;
       i++, packet = packet->copy)
    for (last_jump_packet = find_jump_packet (start_label);
	 last_jump_packet != NULL;
	 last_jump_packet = last_jump_packet->next_jump_packet)
      {
	packet_connect (last_jump_packet->packet, packet->copy, TRUE,
			last_jump_packet->jump_p);
	last_jump_packet->packet = last_jump_packet->packet->copy;
      }
  finish_jumps_table ();
  CLEAR_REG_SET (global_regs_living_after_loop);
  rcsp_free (global_regs_living_after_loop);
  return start_packet;
}

/* The following function set up inheritance relation between LOC and
   already processed locations.  */

static void
set_loc_inheritance (struct loc * loc)
{
  RTX_CODE code = GET_CODE (loc->loc);
  int regno, curr_regno, nregs, curr_nregs;
  struct loc *curr_loc;
  
  if (code == REG && REGNO (loc->loc) < FIRST_PSEUDO_REGISTER)
    {
      regno = REGNO (loc->loc);
      nregs = HARD_REGNO_NREGS (regno, GET_MODE (loc->loc));
      for (curr_loc = hard_reg_loc_chain;
	   curr_loc != NULL;
	   curr_loc = curr_loc->next_hard_reg)
	{
	  curr_regno = REGNO (curr_loc->loc);
	  curr_nregs = HARD_REGNO_NREGS (curr_regno, GET_MODE (curr_loc->loc));
	  if ((regno != curr_regno || nregs != curr_nregs)
	      && ((regno >= curr_regno && regno < curr_regno + curr_nregs)
		  || (curr_regno >= regno && curr_regno < regno + nregs)))
	    {
	      if (regno >= curr_regno
		  && regno + nregs <= curr_regno + curr_nregs)
		loc->supers = new_loc_link (curr_loc, loc->supers);
	      else
		loc->subs = new_loc_link (curr_loc, loc->subs);

	      if (curr_regno >= regno
		  && curr_regno + curr_nregs <= regno + nregs)
		curr_loc->supers = new_loc_link (loc, curr_loc->supers);
	      else
		curr_loc->subs = new_loc_link (loc, curr_loc->subs);
	    }
	}
    }
  else if (code == MEM)
    {
      for (curr_loc = mem_loc_chain;
	   curr_loc != NULL;
	   curr_loc = curr_loc->next_mem)
	if (!mems_in_disjoint_alias_sets_p (loc->loc, curr_loc->loc))
	  {
	    loc->subs = new_loc_link (curr_loc, loc->subs);
	    curr_loc->subs = new_loc_link (loc, curr_loc->subs);
	  }
      all_memory_loc->subs = new_loc_link (loc, all_memory_loc->subs);
      loc->supers = new_loc_link (all_memory_loc, loc->supers);
    }
}

/* The following function returns TRUE if INSN besides (un)conditional
   jump which can be in software pipelined loop.  */

static int
nonjump_p (rtx insn)
{
  /* ??? Now we do not pipeline loops with function calls and asm
     insns (see comments for may_be_pipelined_p).  */
  if (GET_CODE (insn) == JUMP_INSN
      || GET_CODE (insn) == CALL_INSN
      || GET_CODE (PATTERN (insn)) == ASM_INPUT
      || asm_noperands (PATTERN (insn)) >= 0)
    return FALSE;
  return TRUE;
}

/* The following macro defines maximal size (in insns) of loop on
   which RCSP will be tried.  */
#ifndef RCSP_MAX_LOOP_SIZE
#define RCSP_MAX_LOOP_SIZE 20
#endif

/* The following function returns TRUE if we may pipeline the loop
   with body [START_INSN, BOUND_INSN) and FALSE otherwise.  Now there
   are the following constraints for pipelined loops:
     1. Machine does not use cc0.
     2. Loop body insns do not refer for volatile memory.
     3. Loop body insns do not contain asm operands, asm inputs, or
        unspecified volatile operations.
     4. There are no CALL, SETJUMP, computed jumps, and tablejumps in
        the loop body.
     5. There are no exception handlers in the loop.
     6. There is one start point of the loop.
     7. There are not multi-insn sequences (e.g. library calls) in
        the loop.
     8. Maximal loop size (in insns) <= RCSP_MAX_LOOP_SIZE.  */

int
may_be_pipelined_p (rtx start_insn, rtx bound_insn)
{
  rtx insn;
  RTX_CODE code;
  int insns_number = 0;

#ifdef HAVE_cc0
  /* Is it worth to implement software pipelining for such machines???
     If yes, we should implement scheduling a group of insns in the
     software pipeliner.  */
  return FALSE;
#endif
  for (insn = start_insn;
       insn != bound_insn && insn != NULL_RTX;
       insn = NEXT_INSN (insn))
    {
      code = GET_CODE (insn);
      if (code == NOTE)
	{
	  if (NOTE_LINE_NUMBER (insn) == NOTE_INSN_LOOP_BEG
	      || NOTE_LINE_NUMBER (insn) == NOTE_INSN_LOOP_END
	      || NOTE_LINE_NUMBER (insn) == NOTE_INSN_EH_REGION_BEG
	      || NOTE_LINE_NUMBER (insn) == NOTE_INSN_EH_REGION_END
	      || NOTE_LINE_NUMBER (insn) == NOTE_INSN_EXPECTED_VALUE)
	    /* Note setjump ???!!! */
	    return FALSE;
	  else if (NOTE_LINE_NUMBER (insn) > 0
		   || NOTE_LINE_NUMBER (insn) == NOTE_INSN_BASIC_BLOCK
		   || NOTE_LINE_NUMBER (insn) == NOTE_INSN_BLOCK_BEG
		   || NOTE_LINE_NUMBER (insn) == NOTE_INSN_BLOCK_END
		   || NOTE_LINE_NUMBER (insn) == NOTE_INSN_LOOP_CONT
		   || NOTE_LINE_NUMBER (insn) == NOTE_INSN_LOOP_VTOP
		   || NOTE_LINE_NUMBER (insn) == NOTE_INSN_DELETED
		   || NOTE_LINE_NUMBER (insn) == NOTE_INSN_DELETED_LABEL)
	    continue;
	  else
	    abort ();
	}
      else if (code == BARRIER || code == CODE_LABEL)
	continue;
      else if (cond_jump_p (insn) || uncond_jump_p (insn))
	{
	  if (!simplejump_p (insn))
	    insns_number++;
	}
      else if (!nonjump_p (insn))
	return FALSE;
      else
	insns_number++;
      if (GET_RTX_CLASS (code) == 'i'
	  && (volatile_refs_p (PATTERN (insn))
	      /* We don't deal with insn sequences now.  ??? We could
                 implement it in the future.  */
	      || find_reg_note (insn, REG_LIBCALL, NULL_RTX) != NULL_RTX))
	return FALSE;
    }
  if (insn == NULL_RTX && bound_insn != NULL_RTX)
    /* This is weird loop in which bound insn is before start loop
       insn.  Currently we prevent to pipeline such loops.  ??? We
       could pipeline such loop if we correctly find bound insn from
       regions recognized by find_rgns.  */
    return FALSE;
  return insns_number > 0 && insns_number <= RCSP_MAX_LOOP_SIZE;
}

/* The following is start packet and exit packet of unrolled loop.  */
static packet_ptr first_loop_packet;

/* The following is notes which will be placed at the beginning of
   pipelined loop.  */
static rtx start_loop_notes;

#ifndef RCSP_DO_NOT_USE_OBSTACK
/* The most of memory needed for the pipeliner is allocated here.  */
static struct obstack rcsp_memory;
#endif

/* The following function initiates work of the software pipeliner
   with look ahead K (in iterations) on loop with body [START_INSN,
   BOUND_INSN).  */

void
start_RCSP_interface (rtx start_insn, rtx bound_insn, int k)
{
  int i;
  struct loc *curr_loc;

  iteration_look_ahead = k;
  rcsp_debug_level = (sched_verbose == 0 ? 0
		      : sched_verbose < 0 ? -1 : sched_verbose + 1);
#ifndef RCSP_DO_NOT_USE_OBSTACK
  obstack_init (&rcsp_memory);
#endif
  VARRAY_GENERIC_PTR_INIT (insn_reads, 10, "insn_reads");
  VARRAY_GENERIC_PTR_INIT (insn_possible_writes, 10, "insn_possible_writes");
  VARRAY_GENERIC_PTR_INIT (insn_write_barriers, 10, "insn_write_barriers");
  VARRAY_GENERIC_PTR_INIT (insn_writes, 10, "insn_writes");
  insns_length = 0;
  VARRAY_GENERIC_PTR_INIT (insns, 100, "insns");
  start_pipeline_data ();
  first_loop_packet = unroll (start_insn, bound_insn, &start_loop_notes);
  insn_locs_length = insns_length;
  insn_locs = rcsp_malloc (insn_locs_length * sizeof (struct loc *));
  memset ((char *) insn_locs, 0, insn_locs_length * sizeof (struct loc *));
  all_memory_loc = new_loc (gen_rtx_MEM (VOIDmode, gen_rtx_REG (Pmode, 0)),
			    0, 0, NULL);
  hard_reg_loc_chain = mem_loc_chain = NULL;
  /* Create locs.  */
  for (i = 0; i < insns_length; i++)
    if (((insn_ptr) VARRAY_GENERIC_PTR (insns, i))->insn != NULL_RTX)
      process_insn ((insn_ptr) VARRAY_GENERIC_PTR (insns, i));
  /* Make subs and supers.  */
  for (i = 0; i < insn_locs_length; i++)
    for (curr_loc = insn_locs [i];
	 curr_loc != NULL;
	 curr_loc = curr_loc->next_insn_loc)
      set_loc_inheritance (curr_loc);
}

/* The following recursive function generates labels which are in jump
   insns of PACKET and of packets which are achievable from PACKET.
   The function also connects the new labels to corresponding
   insns.  */

static void
add_jump_labels (packet_ptr packet)
{
  op_ptr curr_op;
  packet_ptr target, succ;
  int succ_number;
  rtx nlabel, olabel;

  if (packet->visit_number1 == last_packet_visit_number1)
    return;
  packet->visit_number1 = last_packet_visit_number1;
  for (curr_op = packet->first_op; curr_op != NULL; curr_op = curr_op->next_op)
    if (curr_op->true_target != NULL || curr_op->false_target != NULL)
      {
	target = curr_op->true_target;
	if (cond_branch_p (curr_op->insn) && !curr_op->true_jump_p)
	  target = curr_op->false_target;
	if (target->stop_packet_p
	    && target->first_op->insn->exit_label != NULL_RTX)
	  /* Another branch may be exit from the loop if we permit
             moving one branch insn through another.  */
	  nlabel = target->first_op->insn->exit_label;
	else
	  {
	    nlabel = gen_label_rtx ();
	    target->first_op->insn->notes_before
	      = alloc_EXPR_LIST (REG_SAVE_NOTE, nlabel,
				 target->first_op->insn->notes_before);
	  }
	if (JUMP_LABEL (curr_op->insn->insn) == NULL_RTX)
	  abort ();
	olabel = JUMP_LABEL (curr_op->insn->insn);
	if (!redirect_jump (curr_op->insn->insn, nlabel, FALSE))
	  abort ();
	if (olabel != nlabel)
	  {
	    LABEL_NUSES (olabel)++;
	    LABEL_NUSES (nlabel)--;
	  }
      }
  for (succ_number = 0;
       (succ = packet->succ [succ_number]) != NULL;
       succ_number++)
    add_jump_labels (succ);
}

/* The following variable values is file name and line number
   corresponding to the last output insn in emit_packet.  */
static char *output_last_file;
static int output_last_line; 

/* The following recursive function emits list of NOTES in reverse
   order.  */

static void
emit_notes_list (rtx notes)
{
  if (notes == NULL_RTX)
    return;
  emit_notes_list (XEXP (notes, 1));
  if (GET_CODE (XEXP (notes, 0)) == NOTE)
    {
      rtx note;

      note = emit_note (NOTE_LINE_NUMBER (XEXP (notes, 0)));
      NOTE_SOURCE_FILE (note) = NOTE_SOURCE_FILE (XEXP (notes, 0));
      CODE_LABEL_NUMBER (note) = CODE_LABEL_NUMBER (XEXP (notes, 0));
    }
  else
    emit (XEXP (notes, 0));
}

/* The following recursive function outputs insns of PACKET and all
   packets which are achievable from PACKET.  The function returns
   main loop exit packet if it is achievable from PACKET, NULL
   otherwise.  */

static packet_ptr
emit_packet (packet_ptr packet)
{
  op_ptr curr_op;
  rtx insn;
  packet_ptr succ, pred, result, next_pc_packet, main_exit;
  int succ_number, pred_number;

  if (packet->visit_number1 == last_packet_visit_number1)
    return NULL;
  packet->visit_number1 = last_packet_visit_number1;
  for (curr_op = packet->first_op; curr_op != NULL; curr_op = curr_op->next_op)
    {
      if (curr_op->insn->file != NULL
	  && curr_op->insn->file != NULL
	  && (curr_op->insn->file != output_last_file
	      || curr_op->insn->line != output_last_line))
	{
	  rtx note;

	  output_last_file = curr_op->insn->file;
	  output_last_line = curr_op->insn->line;
	  note = emit_note (output_last_line);
	  NOTE_SOURCE_FILE (note) = output_last_file;
	}
      emit_notes_list (curr_op->insn->notes_before);
      if (curr_op->insn->insn != NULL)
	{
	  /* Output non dummy insn.  */
	  emit (PATTERN (curr_op->insn->insn));
	  if (GET_CODE (curr_op->insn->insn) == JUMP_INSN
	      && JUMP_LABEL (curr_op->insn->insn) != NULL_RTX)
	    {
	      /* Function emit may emit also BARRIER.  */
	      insn = get_last_insn ();
	      if (GET_CODE (insn) == BARRIER)
		insn = prev_nonnote_insn (insn);
	      JUMP_LABEL (insn) = JUMP_LABEL (curr_op->insn->insn);
	    }
	}
      emit_notes_list (curr_op->insn->notes_after);
    }
  result = NULL;
  next_pc_packet = NULL;
  for (succ_number = 0;
       (succ = packet->succ [succ_number]) != NULL;
       succ_number++)
    {
      curr_op = packet_find_branch (packet, succ, TRUE);
      if (curr_op == NULL &&
	  (!succ->stop_packet_p || !succ->main_exit_p))
	{
	  /* Output next pc packet but the exit packet.  */
	  result = emit_packet (succ);
	  next_pc_packet = succ;
	  break;
	}
    }
  for (succ_number = 0;
       (succ = packet->succ [succ_number]) != NULL;
       succ_number++)
    {
      /* Remember: We should output main exit as the last packet.  */
      main_exit = NULL;
      if (succ->stop_packet_p && succ->main_exit_p)
	main_exit = succ;
      else if (succ != next_pc_packet)
	{
	  /* Find pc predecessor of succ.  */
	  for (pred_number = 0;
	       (pred = succ->pred [pred_number]) != NULL;
	       pred_number++)
	    {
	      curr_op = packet_find_branch (pred, succ, TRUE);
	      if (curr_op == NULL)
		/* pred->succ is not a jump.  */
		break;
	    }
	  if (pred == packet)
	    /* We already output the next pc packet.  */
	    abort ();
	  if (pred == NULL)
	    /* There are only jumps onto succ or  */
	    main_exit = emit_packet (succ);
	}
      if (main_exit != NULL)
	{
	  if (result != NULL && result != main_exit)
	    abort ();
	  result = main_exit;
	}
    }
  return result;
}

/* The following function calls pipeliner with DEBUG_FILE.  It should
   be called only after the pipeliner interface initialization.  */

rtx
call_RCSP (FILE * debug_file)
{
  rtx temp, since, first_insn;
  packet_ptr res, main_exit;

  since = rtx_alloc (CONST_INT);
  first_insn = NULL;
  if (first_loop_packet != NULL && first_loop_packet->first_op != NULL)
    first_insn = first_loop_packet->first_op->insn->insn;
  res = pipeline (first_loop_packet, debug_file, ";; ");
  if (res == NULL)
    return NULL;
  start_sequence ();
  last_packet_visit_number1++;
  add_jump_labels (res);
  output_last_file = NULL;
  output_last_line = -1;
  for (; first_insn != NULL_RTX; first_insn = PREV_INSN (first_insn))
    if (GET_CODE (first_insn) == NOTE && NOTE_LINE_NUMBER (first_insn) > 0)
      {
	output_last_line = NOTE_LINE_NUMBER (first_insn);
	output_last_file = (char *) NOTE_SOURCE_FILE (first_insn);
	break;
      }
  emit_note (NOTE_SOFTWARE_PIPELINED_LOOP_BEGIN);
  emit_notes_list (start_loop_notes);
  /* Emit pipelined loop insns.  */
  last_packet_visit_number1++;
  main_exit = emit_packet (res);
  if (main_exit != NULL)
    /* Main exit can be unaccessible for infinite loops.  */
    emit_packet (main_exit);
  emit_note (NOTE_SOFTWARE_PIPELINED_LOOP_END);
  temp = get_insns ();
  end_sequence ();
  return temp;
}

/* The following function finishes work of the software pipeliner on
   the current loop.  */

void
finish_RCSP_interface (void)
{
  int i;

  finish_pipeline_data ();
  free_loc (all_memory_loc);
  for (i = 0; i < insn_locs_length; i++)
    free_insn_locs (insn_locs [i]);
  rcsp_free (insn_locs);
#ifndef RCSP_DO_NOT_USE_OBSTACK
  obstack_free (&rcsp_memory, NULL);
#endif
}



/* This page contains functions of the interface of software pipeliner
   to GCC.  All these functions are called by software pipeliner (the
   abstract data) and should be implemented.  */

/* Allocate memory by RCSP.  */

void *
rcsp_malloc (size_t size)
{
#ifndef RCSP_DO_NOT_USE_OBSTACK
  void *res;

  obstack_blank (&rcsp_memory, size);
  res = obstack_base (&rcsp_memory);
  obstack_finish (&rcsp_memory);
  return res;
#else
  return xmalloc (size);
#endif
}

/* Reallocate memory by RCSP.  */

void *
rcsp_realloc (void * addr, size_t old_size ATTRIBUTE_UNUSED,
	      size_t new_size)
{
#ifndef RCSP_DO_NOT_USE_OBSTACK
  void *res;

  res = rcsp_malloc (new_size);
  memcpy (res, addr, (old_size < new_size ? old_size : new_size));
  return res;
#else
  return xrealloc (addr, new_size);
#endif
}

/* Free memory allocated by RCSP.  */

void
rcsp_free (void * addr ATTRIBUTE_UNUSED)
{
#ifdef RCSP_DO_NOT_USE_OBSTACK
  free (addr);
#endif
}

/* The following function returns TRUE if INSN is a software pipeline
   insn besides (un)conditional branch.  */

int
nonbranch_p (insn_ptr insn)
{
  return nonjump_p (insn->insn);
}

/* The following function returns TRUE if software pipeliner's INSN
   contains a unconditional branch.  */

int
uncond_branch_p (insn_ptr insn)
{
  return uncond_jump_p (insn->insn);
}

/* The following function returns TRUE if software pipeliner's INSN
   contains a conditional branch.  */

int
cond_branch_p (insn_ptr insn)
{
  return cond_jump_p (insn->insn);
}

/* The following function reverses conditional branch INSN, i.e. jump
   and fall trough of the branch is exchanged by reversing the branch
   condition.  The function should make it successfully for branch
   with can_be_converted_p == TRUE.  */

void
convert_cond_branch (insn_ptr insn ATTRIBUTE_UNUSED)
{
  rtx cond ATTRIBUTE_UNUSED;

  /* ??? Now we marks all condtional branch as unconverable.  So we do
     not call this function.  We need implement this function if we
     are going to schedule jumps added to make CFG jump consistent.
     But is it really necessary.  I think the current approach is not
     bad because the added jump with preceding conditional branch
     (this is a frequent case) is usually transformed into one
     conditional branch by jump optimizer.  */
  abort ();
#if 0
  if (!cond_jump_p (insn->insn))
    abort ();
  cond = XEXP (SET_SRC (PATTERN (insn->insn)), 0);
  /* ??? We still need code for change validation.  */
  XEXP (SET_SRC (PATTERN (insn->insn)), 0)
    = gen_rtx (reverse_condition (GET_CODE (cond)), VOIDmode,
	       XEXP (cond, 0), XEXP (cond, 1));
#endif
}

/* The following function returns new unconditional branch onto insn
   TARGET.  The important requiremnt for the branch is that it should
   not create new data dependencies.  */

insn_ptr
create_uncond_branch (insn_ptr target)
{
  rtx label;
  insn_ptr insn;

  if (target->exit_label != NULL_RTX)
    {
      insn = new_insn (make_jump_insn_raw (gen_jump (target->exit_label)),
		       NULL, 0);
      JUMP_LABEL (insn->insn) = target->exit_label;
    }
  else
    {
      /* We generate labels for such jumps in add_jump_labels.  */
      label = gen_label_rtx ();
      insn = new_insn (make_jump_insn_raw (gen_jump (label)), NULL, 0);
      JUMP_LABEL (insn->insn) = label;
    }
  return insn;
}

/* The following function returns TRUE if we need jump onto PACKET even
   there is only one packet predecessor without unconditional jump.  Now
   this is true for all loop exits.  We make the jump even for main loop exit
   (which correspond the first insn after the loop) although jump in
   this case is not necessary.  Fortunately, such jump is removed by
   jump optimizer.  */

int
dest_packet_requires_jump_p (packet_ptr packet)
{
  if (packet->first_op->insn->exit_label != NULL_RTX)
    {
      /* Here we can say that insn is in a stop packet.  */
      if (packet->first_op->insn->insn != NULL)
	/* It should be non-dummy insn in a stop packet.  */
	abort ();
      return TRUE;
    }
  return FALSE;
}

/* The following function returns TRUE if INSN1 and INSN2 are the same
   insn possibly from different iterations.  */

int
insn_eq_p (insn_ptr insn1, insn_ptr insn2)
{
  return insn1->orig_insn == insn2->orig_insn;
}

/* The folowing function returns hash code of INSN.  The same insns
   from different iterations have the same hash code.  */

unsigned
insn_hash (insn_ptr insn)
{
  return (unsigned) insn->orig_insn;
}

/* The following function returns (compact if DENSE_P) symbolic INSN
   representation.  For brevity we simply return UID of INSN.  The
   representation stores in static memory so the next function call
   may rewrite it.  */

const char *
insn_name (insn_ptr insn, int dense_p)
{
  if (insn == NULL || insn->insn == NULL_RTX)
    return (dense_p ? "dummy" : "dummy op");
  else
    {
      static char name [60];
      char buf[2048];

      print_insn (buf, insn->orig_insn, 0);
      buf[40] = 0;
      sprintf (name, (dense_p ? "%s" : "%s"), buf);
      return name;
    }
}

/* The following function returns TRUE if it is dummy insn (which
   should be ignored in many cases) for software pipeliner.  Dummy
   insns are placed in stop packets and in the root packet.  */

int
insn_dummy_p (insn_ptr insn)
{
  return insn == NULL || insn->insn == NULL_RTX;
}

/* The following function makes and returns copy of RCSP INSN.  */

insn_ptr
insn_copy (insn_ptr insn)
{
  insn_ptr copy;
  rtx insn_copy;
  
  copy = new_insn (insn->orig_insn, insn->file, insn->line);
  if (insn->notes_before != NULL_RTX)
    copy->notes_before = copy_rtx (insn->notes_before);
  if (insn->notes_after != NULL_RTX)
    copy->notes_after = copy_rtx (insn->notes_after);
  copy->live_reg_set_p = insn->live_reg_set_p;
  if (GET_CODE (insn->insn) == JUMP_INSN)
    {
      insn_copy = make_jump_insn_raw (copy_insn (PATTERN (insn->insn)));
      JUMP_LABEL (insn_copy) = JUMP_LABEL (insn->insn);
      remove_reg_br_prob (insn_copy);
    }
  else
    insn_copy = make_insn_raw (copy_insn (PATTERN (insn->insn)));
  PREV_INSN (insn_copy) = NEXT_INSN (insn_copy) = NULL_RTX;
  copy->insn = insn_copy;
  return copy;
}

/* The following function returns latency time on dependency SRC->DEST
   of type DEP_MODE.  */

int
insn_dep_time (insn_ptr src, insn_ptr dest, int dep_mode)
{
  recog_memoized (src->insn);
  recog_memoized (dest->insn);

  if (INSN_CODE (src->insn) < 0 || INSN_CODE (dest->insn) < 0)
    return 0;
  if (dep_mode == USE_DEF)
    return 0;
  else if (dep_mode == DEF_USE)
    return (bypass_p (src->insn)
	    ? insn_latency (src->insn, dest->insn)
	    : insn_default_latency (src->insn));
  else if (dep_mode != DEF_DEF)
    abort ();
  else
    {
      int cost
	= insn_default_latency (src->insn) - insn_default_latency (dest->insn);

      return (cost < 0 ? 0 : cost);
    }
}

/* The following function returns TRUE if speculative move INSN into
   PACKET is risky.  In other words such move is prohibited.  ??? Now
   we get the information only from insn itself.  We could use
   information about context too (e.g. all path from PACKET contains
   load of the same location as INSN does).  */

int
insn_risky_speculative_move_p (insn_ptr insn, packet_ptr packet ATTRIBUTE_UNUSED)
{
  int insn_class;

  if (insn == NULL || insn->insn == NULL)
    abort ();

  if (insn->live_reg_set_p)
    return TRUE;

  insn_class = haifa_classify_insn (insn->insn);

  /* Handle non-load insns.  */
  switch (insn_class)
    {
    case TRAP_FREE:
      return FALSE;
    case TRAP_RISKY:
      return TRUE;
    default:;
    }

  /* Handle loads.  */
  if (!flag_schedule_speculative_load)
    return TRUE;
  switch (insn_class)
    {
    case IFREE:
      return FALSE;
    case IRISKY:
      return TRUE;
    default:;
    }
  return TRUE;
}

/* The following function initializes CPU_STATE as if all functional
   units were free.  */

void
cpu_state_reset (cpu_state_ptr cpu_state)
{
  state_reset (cpu_state);
}

/* The following function returns FALSE if INSN can not be issued in
   this CPU_STATE.  Otherwise, the function returns TRUE and changes
   CPU_STATE as if INSN were issued.  */

int
cpu_state_issue (insn_ptr insn, cpu_state_ptr cpu_state)
{
  int delay;
#if FIRST_CYCLE_MULTIPASS_SCHEDULING && defined (SCHEDULER_BUBBLE)
  int i;
  rtx bubble;
  state_t temp_state = alloca (dfa_state_size);
#endif

  recog_memoized (insn->insn);
  if (INSN_CODE (insn->insn) < 0)
    return 1;
  delay = state_transition (cpu_state, insn->insn);
  if ( delay < 0)
    return 1;
  else if (delay > 0)
    return 0;
#if FIRST_CYCLE_MULTIPASS_SCHEDULING && defined (SCHEDULER_BUBBLE)
  for (i = 0;
       (bubble = SCHEDULER_BUBBLE (i)) != NULL_RTX;
       i++)
    {
      memcpy (temp_state, cpu_state, dfa_state_size);
      if (state_transition (temp_state, bubble) < 0
	  && state_transition (temp_state, insn->insn) < 0)
	break;
    }
  if (bubble != NULL_RTX)
    {
      memcpy (cpu_state, temp_state, dfa_state_size);
      return 1;
    }
  else
#endif
    return 0;
}

/* The following function changes CPU_STATE for cpu cycle tick.  */

void
cpu_state_advance_cycle (cpu_state_ptr cpu_state)
{
#ifdef DFA_SCHEDULER_PRE_CYCLE_INSN
  state_transition (cpu_state, DFA_SCHEDULER_PRE_CYCLE_INSN);
#endif
  state_transition (cpu_state, NULL);
#ifdef DFA_SCHEDULER_POST_CYCLE_INSN
  state_transition (cpu_state, DFA_SCHEDULER_POST_CYCLE_INSN);
#endif
}

/* The following function returns size of data which represents all
   current and projected functional unit reservations.  This is simply
   DFA-based pipeline description state.  */

int
cpu_state_size (void)
{
  return state_size ();
}

/* The following function makes union CPU_STATE and CPU_STATE2.  The
   result will be placed in CPU_STATE.  */

void
cpu_state_union (cpu_state_ptr cpu_state1 ATTRIBUTE_UNUSED,
		 cpu_state_ptr cpu_state2 ATTRIBUTE_UNUSED)
{
  /* ??? Now we have no implementation of this in our DFA-based
     pipeline description.  Such implementation could result in much
     bigger automata.  So another solution could be implementation of
     CPU state as a list of automaton states.  For the most loops the
     lists will be short because merging frontier packets is not
     frequent event.  */
}

/* The following function returns TRUE if LOC1 refers for the same
   location as LOC2. ??? We need better recognition of aliases.  */

int
loc_eq_p (loc_ptr loc1, loc_ptr loc2)
{
  if (GET_CODE (loc1->loc) != GET_CODE (loc2->loc))
    return FALSE;
  if (GET_CODE (loc1->loc) == REG)
    {
      if (REGNO (loc1->loc) != REGNO (loc2->loc))
	return FALSE;
      if (REGNO (loc1->loc) >= FIRST_PSEUDO_REGISTER)
	return TRUE;
      return GET_MODE (loc1->loc) == GET_MODE (loc2->loc);
    }
    
  if (GET_CODE (loc1->loc) != MEM)
    abort ();
  return loc1->loc == loc2->loc;
}

/* The following function returns hash code of location LOC.  */

unsigned
loc_hash (loc_ptr loc)
{
  if (GET_CODE (loc->loc) == REG)
    {
      if (REGNO (loc->loc) >= FIRST_PSEUDO_REGISTER)
	return (unsigned) REGNO (loc->loc);
      else
	return ((unsigned) REGNO (loc->loc)
		+ ((unsigned) GET_MODE (loc->loc) << 16));
    }

  if (GET_CODE (loc->loc) != MEM)
    abort ();
  return (unsigned) loc->loc;
}

/* The following function returns the first (if FIRST_P) or the next
   (otherwise) sub-location (direct or indirect) of LOC.  If there are
   no more sub-locations the function returns NULL.  */

loc_ptr
loc_sub (loc_ptr loc, int first_p)
{
  static struct loc_link *curr_link = NULL;

  if (first_p)
    curr_link = loc->subs;
  else if (curr_link != NULL)
    curr_link = curr_link->next;
  if (curr_link == NULL)
    return NULL;
  else
    return curr_link->loc;
}

/* The following function returns the first (if FIRST_P) or the next
   (otherwise) super-location (direct or indirect) of LOC.  If there
   are no more super-locations, the function returns NULL.  */

loc_ptr
loc_super (loc_ptr loc, int first_p)
{
  static struct loc_link *curr_link;

  if (first_p)
    curr_link = loc->supers;
  else if (curr_link != NULL)
    curr_link = curr_link->next;
  if (curr_link == NULL)
    return NULL;
  else
    return curr_link->loc;
}

/* The following function returns N-th locations of INSN.  Flags of
   that INSN uses the returned location is passed through USAGE_P.  If
   the location is written, type of writing is returned through
   WRITE_KIND.  Used locations should be returned the first.  */

loc_ptr
getloc (insn_ptr insn, int n, int * usage_p, enum write_kind * write_kind)
{
  /* Dummy operation does not refer for locations.  */
  if (insn == NULL || insn->insn == NULL_RTX)
    abort ();
  if (n == 0)
    {
      struct loc *curr_loc;

      insn_reads_length = insn_possible_writes_length
	= insn_write_barriers_length = insn_writes_length = 0;
      for (curr_loc = insn_locs [insn->number];
	   curr_loc != NULL;
	   curr_loc = curr_loc->next_insn_loc)
	if (curr_loc->usage_p)
	  add_vector_element (&insn_reads, curr_loc, &insn_reads_length);
	else if (curr_loc->write_kind == USUAL_WRITE)
	  add_vector_element (&insn_writes, curr_loc, &insn_writes_length);
	else if (curr_loc->write_kind == POSSIBLE_WRITE)
	  add_vector_element (&insn_possible_writes, curr_loc,
			      &insn_possible_writes_length);
	else if (curr_loc->write_kind != WRITE_BARRIER)
	  abort ();
	else
	  add_vector_element (&insn_write_barriers, curr_loc,
			      &insn_write_barriers_length);
    }

  *write_kind = 0;
  *usage_p = FALSE;
  if (n < insn_reads_length)
    {
      *usage_p = TRUE;
      return (loc_ptr) VARRAY_GENERIC_PTR (insn_reads, n);
    }
  else if ((n -= insn_reads_length) < insn_possible_writes_length)
    {
      *write_kind = 1;
      return (loc_ptr) VARRAY_GENERIC_PTR (insn_possible_writes, n);
    }
  else if ((n -= insn_possible_writes_length) < insn_writes_length)
    return (loc_ptr) VARRAY_GENERIC_PTR (insn_writes, n);
  else if ((n -= insn_writes_length) < insn_write_barriers_length)
    {
      *write_kind = 2;
      return (loc_ptr) VARRAY_GENERIC_PTR (insn_write_barriers, n);
    }
  else
    return NULL;
}
#endif /* RCSP_SOFTWARE_PIPELINING */
