/*
   (c) Copyright 2002-2003  Denis Oliver Kropp <dok@directfb.org>
   All rights reserved.

   XDirectFB is mainly based on XDarwin and
   also contains some KDrive, XFree and XWin code.
*/
/*
 *Copyright (C) 1994-2000 The XFree86 Project, Inc. All Rights Reserved.
 *
 *Permission is hereby granted, free of charge, to any person obtaining
 * a copy of this software and associated documentation files (the
 *"Software"), to deal in the Software without restriction, including
 *without limitation the rights to use, copy, modify, merge, publish,
 *distribute, sublicense, and/or sell copies of the Software, and to
 *permit persons to whom the Software is furnished to do so, subject to
 *the following conditions:
 *
 *The above copyright notice and this permission notice shall be
 *included in all copies or substantial portions of the Software.
 *
 *THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 *EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 *MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 *NONINFRINGEMENT. IN NO EVENT SHALL THE XFREE86 PROJECT BE LIABLE FOR
 *ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF
 *CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
 *WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 *
 *Except as contained in this notice, the name of the XFree86 Project
 *shall not be used in advertising or otherwise to promote the sale, use
 *or other dealings in this Software without prior written authorization
 *from the XFree86 Project.
 *
 * Authors:	Harold L Hunt II
 */
/* $XFree86: xc/programs/Xserver/hw/xwin/winclipboardinit.c,v 1.1 2003/02/12 15:01:38 alanh Exp $ */

/* Standard library headers */
#include <assert.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <setjmp.h>
#include <pthread.h>

/* X headers */
#include "X.h"
#include "Xatom.h"
#include "Xproto.h"
#include "Xutil.h"
#include "Xlocale.h"

#include "directfbClipboard.h"
#include "directfbPasteboard.h"
#include "directfbX.h"

#define SELECTION_DEBUG 0

#if SELECTION_DEBUG
#define DEBUG(x...) fprintf(stderr,x)
#else
#define DEBUG(x...) do {} while (0)
#endif

/*
 * Global variables
 */

static jmp_buf g_jmpEntry;

#define XDIRECTFB_JMP_OKAY              0
#define XDIRECTFB_JMP_ERROR_IO          2

#define XDIRECTFB_CONNECT_RETRIES       3
#define XDIRECTFB_CONNECT_DELAY         4

/*
 * References to external symbols
 */

extern char *display;

/******************************************************************************/

static void *XDirectFBClipboardProc (void *pArg);

/*
 * Intialize the Clipboard module
 */

Bool
XDirectFBClipboardInit (ScreenPtr pScreen)
{
     DirectFBScreenPtr pScreenPriv = DIRECTFB_PRIV(pScreen);

     DEBUG ("XDirectFBClipboardInit ()\n");

     /* Spawn a thread for the Clipboard module */
     if (pthread_create (&pScreenPriv->clipboardThread, NULL,
                         XDirectFBClipboardProc, pScreen)) {
          /* Bail if thread creation failed */
          ErrorF ("XDirectFBClipboardInit - pthread_create failed.\n");
          return FALSE;
     }

     return TRUE;
}

/******************************************************************************/

static Bool ReadClipboard(char **data, int *len)
{
     char         *mime;
     void         *buf;
     char         *buf2;
     unsigned int  size;

     if (dfb->GetClipboardData (dfb, &mime, &buf, &size))
          return FALSE;

     if (strcmp (mime, "text/plain")) {
          free (mime);
          free (buf);

          return FALSE;
     }

     buf2 = malloc( size + 1 );
     if (!buf2) {
          free (mime);
          free (buf);

          return FALSE;
     }

     memcpy( buf2, buf, size );

     buf2[size] = 0;

     free (buf);

     *data = buf2;
     *len  = size + 1;

     free (mime);

     return TRUE;
}

/*
 * XDirectFBClipboardErrorHandler - Our application specific error handler
 */

static int
XDirectFBClipboardErrorHandler (Display *pDisplay, XErrorEvent *pErr)
{
     char pszErrorMsg[100];

     XGetErrorText (pDisplay,
                    pErr->error_code,
                    pszErrorMsg,
                    sizeof (pszErrorMsg));
     ErrorF ("XDirectFBClipboardErrorHandler - ERROR: \n\t%s\n", pszErrorMsg);

     if (pErr->error_code==BadWindow
         || pErr->error_code==BadMatch
         || pErr->error_code==BadDrawable) {
          pthread_exit (NULL);
     }

     pthread_exit (NULL);

     return 0;
}


/*
 * XDirectFBClipboardIOErrorHandler - Our application specific IO error handler
 */

static int
XDirectFBClipboardIOErrorHandler (Display *pDisplay)
{
     fprintf (stderr, "\nXDirectFBClipboardIOErrorHandler!\n\n");

     /* Restart at the main entry point */
     longjmp (g_jmpEntry, XDIRECTFB_JMP_ERROR_IO);

     return 0;
}

/*
 * Main thread function
 */

static void *
XDirectFBClipboardProc (void *pArg)
{
     Atom                 atomLocalProperty, atomCompoundText;
     Atom                 atomUTF8String, atomTargets;
     int                  iReturn;
     Display             *pDisplay;
     Window               iWindow;
     Atom                 atomDeleteWindow;
     int                  iRetries;
     char                 szDisplay[512];
     XTextProperty        xtpText;
     XEvent               event;
     XSelectionEvent      eventSelection;
     unsigned long        ulReturnBytesLeft;
     XICCEncodingStyle    xiccesStyle;
     int                  iUTF8;
     char                *pszUTF8 = NULL;
     char                *pszTextList[2];
     int                  iCount;
     char               **ppszTextList = NULL;
     struct timeval       stamp;
     
     ScreenPtr            pScreen     = pArg;
     DirectFBScreenPtr    pScreenPriv = DIRECTFB_PRIV(pScreen);


     DEBUG ("XDirectFBClipboardProc - Hello\n");

     DEBUG ("XDirectFBClipboardProc - Calling pthread_mutex_lock ()\n");

     /* Grab our garbage mutex to satisfy pthread_cond_wait */
     iReturn = pthread_mutex_lock (&pScreenPriv->serverStarted);
     if (iReturn != 0) {
          ErrorF ("XDirectFBClipboardProc - pthread_mutex_lock () failed: %d\n",
                  iReturn);
          pthread_exit (NULL);
     }

     DEBUG ("XDirectFBClipboardProc - pthread_mutex_lock () returned.\n");

     /* Release the garbage mutex */
     pthread_mutex_unlock (&pScreenPriv->serverStarted);

     DEBUG ("XDirectFBClipboardProc - pthread_mutex_unlock () returned.\n");

     /* Allow multiple threads to access Xlib */
     if (XInitThreads () == 0) {
          ErrorF ("XDirectFBClipboardProc - XInitThreads failed.\n");
          pthread_exit (NULL);
     }

     DEBUG ("XDirectFBClipboardProc - XInitThreads () returned.\n");

     /* Set jump point for Error exits */
     iReturn = setjmp (g_jmpEntry);

     /* Check if we should continue operations */
     if (iReturn != XDIRECTFB_JMP_ERROR_IO
         && iReturn != XDIRECTFB_JMP_OKAY) {
          /* setjmp returned an unknown value, exit */
          ErrorF ("XDirectFBClipboardProc - setjmp returned: %d exiting\n",
                  iReturn);
          pthread_exit (NULL);
     }
     else if (iReturn == XDIRECTFB_JMP_ERROR_IO) {
          DEBUG ("XDirectFBClipboardProc - setjmp returned\n");
     }

     /* Initialize retry count */
     iRetries = 0;

     /* Setup the display connection string x */
     snprintf (szDisplay, 512, ":%s.%d", display, pScreen->myNum);

     /* Print the display connection string */
     DEBUG ("XDirectFBClipboardProc - DISPLAY=%s\n", szDisplay);

     /* Open the X display */
     do {
          pDisplay = XOpenDisplay (szDisplay);
          if (pDisplay == NULL) {
               ErrorF ("XDirectFBClipboardProc - Could not open display, "
                       "try: %d, sleeping: %d\n",
                       iRetries + 1, XDIRECTFB_CONNECT_DELAY);
               ++iRetries;
               sleep (XDIRECTFB_CONNECT_DELAY);
               continue;
          }
          else
               break;
     }
     while (pDisplay == NULL && iRetries < XDIRECTFB_CONNECT_RETRIES);

     /* Make sure that the display opened */
     if (pDisplay == NULL) {
          ErrorF ("XDirectFBClipboardProc - Failed opening the display, giving up\n");
          pthread_exit (NULL);
     }

     pScreenPriv->pDisplay = pDisplay;

     DEBUG ("XDirectFBClipboardProc - XOpenDisplay () returned and "
             "successfully opened the display.\n");

     /* Select event types to watch */
     if (XSelectInput (pDisplay,
                       DefaultRootWindow (pDisplay),
                       SubstructureNotifyMask |
                       StructureNotifyMask |
                       PropertyChangeMask) == BadWindow)
          ErrorF ("XDirectFBClipboardProc - XSelectInput generated BadWindow "
                  "on RootWindow\n\n");

     /* Create a messaging window */
     iWindow = XCreateSimpleWindow (pDisplay,
                                    DefaultRootWindow (pDisplay),
                                    1, 1,
                                    500, 500,
                                    0,
                                    BlackPixel (pDisplay, 0),
                                    BlackPixel (pDisplay, 0));
     if (iWindow == 0) {
          ErrorF ("XDirectFBClipboardProc - Could not create a window\n");
          pthread_exit (NULL);
     }

     pScreenPriv->clipboardWindow = iWindow;

     /* This looks like our only hope for getting a message before shutdown */
     /* Register for WM_DELETE_WINDOW message from window manager */
     atomDeleteWindow = XInternAtom (pDisplay, "WM_DELETE_WINDOW", False);
     XSetWMProtocols (pDisplay, iWindow, &atomDeleteWindow, 1);

     /* Set error handler */
     XSetErrorHandler (XDirectFBClipboardErrorHandler);
     XSetIOErrorHandler (XDirectFBClipboardIOErrorHandler);

     /* Local property to hold pasted data */
     atomLocalProperty = XInternAtom (pDisplay, "XDIRECTFB_CUT_BUFFER", False);
     if (atomLocalProperty == None) {
          ErrorF ("XDirectFBClipboardProc - Could not create XDIRECTFB_CUT_BUFFER atom\n");
          pthread_exit (NULL);
     }

     pScreenPriv->atomLocalProperty = atomLocalProperty;

     /* Create an atom for UTF8_STRING */
     atomUTF8String = XInternAtom (pDisplay, "UTF8_STRING", False);
     if (atomUTF8String == None) {
          ErrorF ("XDirectFBClipboardProc - Could not create UTF8_STRING atom\n");
          pthread_exit (NULL);
     }

     /* Create an atom for COMPOUND_TEXT */
     atomCompoundText = XInternAtom (pDisplay, "COMPOUND_TEXT", False);
     if (atomCompoundText == None) {
          ErrorF ("XDirectFBClipboardProc - Could not create COMPOUND_TEXT atom\n");
          pthread_exit (NULL);
     }

     pScreenPriv->atomCompoundText = atomCompoundText;
     
     /* Create an atom for TARGETS */
     atomTargets = XInternAtom (pDisplay, "TARGETS", False);
     if (atomTargets == None) {
          ErrorF ("XDirectFBClipboardProc - Could not create TARGETS atom\n");
          pthread_exit (NULL);
     }

     while (1) {
          /* Get the next event */
          XNextEvent (pDisplay, &event);

          /* Branch on the event type */
          switch (event.type) {
               case ClientMessage:
                    if (event.xclient.data.l[0] == atomDeleteWindow) {
                         DEBUG ("\nReceived WM_DELETE_WINDOW\n\n");
                         return 0;
                    }
                    else
                         DEBUG ("\nUnknown ClientMessage\n\n");
                    break;

               case SelectionClear:
                    DEBUG ("SelectionClear\n");
                    break;


                    /*
                     * SelectionRequest
                     */

               case SelectionRequest:
                    DEBUG ("SelectionRequest\n");
#if SELECTION_DEBUG
                    {
                         char              *pszAtomName = NULL;

                         DEBUG ("SelectionRequest - target %lu\n",
                                 event.xselectionrequest.target);

                         pszAtomName = XGetAtomName (pDisplay,
                                                     event.xselectionrequest.target);
                         DEBUG ("SelectionRequest - Target atom name %s\n", pszAtomName);
                         XFree (pszAtomName);
                         pszAtomName = NULL;
                    }
#endif

                    /* Abort if invalid target type */
                    if (event.xselectionrequest.target != XA_STRING
                        && event.xselectionrequest.target != atomUTF8String
                        && event.xselectionrequest.target != atomCompoundText
                        && event.xselectionrequest.target != atomTargets) {
                         DEBUG("SelectionRequest - unknown target!\n");
                         /* Setup selection notify event */
                         eventSelection.type = SelectionNotify;
                         eventSelection.serial = event.xselectionrequest.serial;
                         eventSelection.send_event = True;
                         eventSelection.display = pDisplay;
                         eventSelection.requestor = event.xselectionrequest.requestor;
                         eventSelection.selection = event.xselectionrequest.selection;
                         eventSelection.target = event.xselectionrequest.target;
                         eventSelection.property = None;
                         eventSelection.time = event.xselectionrequest.time;

                         /* Notify the requesting window that the operation is complete */
                         iReturn = XSendEvent (pDisplay,
                                               eventSelection.requestor,
                                               False,
                                               0L,
                                               (XEvent *) &eventSelection);
                         if (iReturn == BadValue || iReturn == BadWindow) {
                              ErrorF ("SelectionRequest - XSendEvent () failed\n");
                              pthread_exit (NULL);
                         }

                         break;
                    }

                    /* Handle targets type of request */
                    if (event.xselectionrequest.target == atomTargets) {
                         Atom atomTargetArr[4] = {atomTargets,
                              atomCompoundText,
                              atomUTF8String,
                              XA_STRING};

                         /* Try to change the property */
                         iReturn = XChangeProperty (pDisplay,
                                                    event.xselectionrequest.requestor,
                                                    event.xselectionrequest.property,
                                                    event.xselectionrequest.target,
                                                    8,
                                                    PropModeReplace,
                                                    (char *) atomTargetArr,
                                                    sizeof (atomTargetArr));
                         if (iReturn == BadAlloc
                             || iReturn == BadAtom
                             || iReturn == BadMatch
                             || iReturn == BadValue
                             || iReturn == BadWindow) {
                              ErrorF ("SelectionRequest - XChangeProperty failed: %d\n",
                                      iReturn);
                         }

                         /* Setup selection notify xevent */
                         eventSelection.type = SelectionNotify;
                         eventSelection.send_event = True;
                         eventSelection.display   = pDisplay;
                         eventSelection.requestor = event.xselectionrequest.requestor;
                         eventSelection.selection = event.xselectionrequest.selection;
                         eventSelection.target    = event.xselectionrequest.target;
                         eventSelection.property  = event.xselectionrequest.property;
                         eventSelection.time      = event.xselectionrequest.time;

                         /*
                          * Notify the requesting window that
                          * the operation has completed
                          */
                         iReturn = XSendEvent (pDisplay,
                                               eventSelection.requestor,
                                               False,
                                               0L,
                                               (XEvent *) &eventSelection);
                         if (iReturn == BadValue || iReturn == BadWindow) {
                              ErrorF ("SelectionRequest - XSendEvent () failed\n");
                         }
                         break;
                    }

                    /* Setup the string style */
                    if (event.xselectionrequest.target == XA_STRING)
                         xiccesStyle = XStringStyle;
                    else if (event.xselectionrequest.target == atomUTF8String)
                         xiccesStyle = XUTF8StringStyle;
                    else if (event.xselectionrequest.target == atomCompoundText)
                         xiccesStyle = XCompoundTextStyle;
                    else
                         xiccesStyle = XStringStyle;

                    /* Get a pointer to the clipboard text */
                    if (!ReadClipboard( &pszUTF8, &iUTF8 )) {
                         pszUTF8 = strdup("");
                         iUTF8   = 2;
                    }

                    if (event.xselectionrequest.target == atomUTF8String) {
                         /* Initialize the text property */
                         xtpText.value  = strdup (pszUTF8);
                         xtpText.nitems = iUTF8;
                    }
                    else {
                         /* Setup our text list */
                         pszTextList[0] = pszUTF8;
                         pszTextList[1] = NULL;
     
                         /* Initialize the text property */
                         xtpText.value = NULL;
     
                         /* Create the text property from the text list */
                         iReturn = Xutf8TextListToTextProperty (pDisplay,
                                                                pszTextList,
                                                                1,
                                                                xiccesStyle,
                                                                &xtpText);
                         if (iReturn == XNoMemory || iReturn == XLocaleNotSupported) {
                              ErrorF ("SelectionRequest - Xutf8TextListToTextProperty "
                                      "failed: %d\n",
                                      iReturn);
                              pthread_exit (NULL);
                         }
                    }

                    /* Copy the clipboard text to the requesting window */
                    iReturn = XChangeProperty (pDisplay,
                                               event.xselectionrequest.requestor,
                                               event.xselectionrequest.property,
                                               event.xselectionrequest.target,
                                               8,
                                               PropModeReplace,
                                               xtpText.value,
                                               xtpText.nitems);
                    if (iReturn == BadAlloc || iReturn == BadAtom
                        || iReturn == BadMatch || iReturn == BadValue
                        || iReturn == BadWindow) {
                         ErrorF ("SelectionRequest - XChangeProperty failed: %d\n",
                                 iReturn);
                         pthread_exit (NULL);
                    }

                    if (xtpText.value) {
                         free (xtpText.value);
                         xtpText.value = NULL;
                    }

                    free( pszUTF8 );

                    /* Setup selection notify event */
                    eventSelection.type = SelectionNotify;
                    eventSelection.serial = event.xselectionrequest.serial;
                    eventSelection.send_event = True;
                    eventSelection.display = pDisplay;
                    eventSelection.requestor = event.xselectionrequest.requestor;
                    eventSelection.selection = event.xselectionrequest.selection;
                    eventSelection.target = event.xselectionrequest.target;
                    eventSelection.property = event.xselectionrequest.property;
                    eventSelection.time = event.xselectionrequest.time;

                    /* Notify the requesting window that the operation has completed */
                    iReturn = XSendEvent (pDisplay,
                                          eventSelection.requestor,
                                          False,
                                          0L,
                                          (XEvent *) &eventSelection);
                    if (iReturn == BadValue || iReturn == BadWindow) {
                         ErrorF ("SelectionRequest - XSendEvent () failed\n");
                         pthread_exit (NULL);
                    }
                    break;


                    /*
                     * SelectionNotify
                     */ 

               case SelectionNotify:
                    DEBUG ("SelectionNotify\n");
#if SELECTION_DEBUG
                    {
                         char       *pszAtomName;

                         pszAtomName = XGetAtomName (pDisplay,
                                                     event.xselection.selection);

                         DEBUG ("SelectionNotify - ATOM: %s\n",
                                 pszAtomName);

                         XFree (pszAtomName);
                    }
#endif

                    /*
                     *
                     * What are we doing here?
                     *
                     */
                    if (event.xselection.property == None) {
                         if (event.xselection.target == XA_STRING) {
                              DEBUG ("SelectionNotify XA_STRING\n");
                              break;
                         }
                         else if (event.xselection.target == atomUTF8String) {
                              DEBUG ("SelectionNotify UTF8\n");
                              iReturn = XConvertSelection (pDisplay,
                                                           event.xselection.selection,
                                                           XA_STRING,
                                                           atomLocalProperty,
                                                           iWindow,
                                                           CurrentTime);
                              if (iReturn == BadAtom || iReturn == BadWindow) {
                                   ErrorF ("SelectionNotify - XConvertSelection () "
                                           "failed\n");
                                   pthread_exit (NULL);
                              }
                              break;
                         }
                         else if (event.xselection.target == atomCompoundText) {
                              DEBUG ("SelectionNotify CompoundText\n");
                              iReturn = XConvertSelection (pDisplay,
                                                           event.xselection.selection,
                                                           atomUTF8String,
                                                           atomLocalProperty,
                                                           iWindow,
                                                           CurrentTime);
                              if (iReturn == BadAtom || iReturn == BadWindow) {
                                   ErrorF ("SelectionNotify - XConvertSelection () "
                                           "failed\n");
                                   pthread_exit (NULL);
                              }
                              break;
                         }
                         else {
                              DEBUG("Unknown format\n");
                              break;
                         }
                    }
                    
                    /* Retrieve the size of the stored data */
                    iReturn = XGetWindowProperty (pDisplay,
                                                  iWindow,
                                                  atomLocalProperty,
                                                  0,
                                                  0, /* Don't get data, just size */
                                                  False,
                                                  AnyPropertyType,
                                                  &xtpText.encoding,
                                                  &xtpText.format,
                                                  &xtpText.nitems,
                                                  &ulReturnBytesLeft,
                                                  &xtpText.value);
                    if (iReturn != Success) {
                         ErrorF ("SelectionNotify - XGetWindowProperty () failed\n");
                         pthread_exit (NULL);
                    }

                    DEBUG ("SelectionNotify - returned data %lu left %lu\n",
                            xtpText.nitems, ulReturnBytesLeft);
                    
                    /* Request the selection data */
                    iReturn = XGetWindowProperty (pDisplay,
                                                  iWindow,
                                                  atomLocalProperty,
                                                  0,
                                                  ulReturnBytesLeft,
                                                  True,
                                                  AnyPropertyType,
                                                  &xtpText.encoding,
                                                  &xtpText.format,
                                                  &xtpText.nitems,
                                                  &ulReturnBytesLeft,
                                                  &xtpText.value);
                    if (iReturn != Success) {
                         ErrorF ("SelectionNotify - XGetWindowProperty () failed\n");
                         pthread_exit (NULL);
                    }

                    /* Convert the text property to a text list */
                    Xutf8TextPropertyToTextList (pDisplay,
                                                 &xtpText,
                                                 &ppszTextList,
                                                 &iCount);
                    if (!iCount) {
                         ErrorF ("%s: Could not convert text property\n",
                                 __FUNCTION__);
                         break;
                    }

                    dfb->SetClipboardData (dfb, "text/plain",
                                           ppszTextList[0],
                                           strlen(ppszTextList[0]), &stamp);

                    pScreenPriv->clipboardStamp = (stamp.tv_sec * 1000) +
                                                  (stamp.tv_usec / 1000);

                    DEBUG ("SelectionNotify - got '%s'\n", ppszTextList[0]);
                    
                    /* Free the data returned from XGetWindowProperty */
                    XFreeStringList (ppszTextList);
                    XFree (xtpText.value);
                    
                    break;

               default:
                    break;
          }
     }

     return 0;
}

