/* atomic.h: atomic operation emulation for FR-V
 *
 * Copyright (C) 2003 Red Hat, Inc. All Rights Reserved.
 * Written by David Howells (dhowells@redhat.com)
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version
 * 2 of the License, or (at your option) any later version.
 */
#ifndef _ASM_ATOMIC_H
#define _ASM_ATOMIC_H

#include <linux/config.h>
#include <asm/spr-regs.h>

#ifdef CONFIG_SMP
#error not SMP safe
#endif

extern unsigned int __atomic_lock;

#define __atomic_begin(flags)				\
do {							\
	unsigned long npsr;				\
	asm volatile("	movsg	psr,%0		\n"	\
		     "	ori	%0,%2,%1	\n"	\
		     "	movgs	%1,psr		\n"	\
		     : "=r"(flags), "=r"(npsr)		\
		     : "i" (PSR_PIL)			\
		     : "memory");			\
} while(0)

#define __atomic_end(flags)				\
do {							\
	asm volatile("	membar			\n"	\
		     "	movgs	%0,psr		\n"	\
		     :					\
		     : "r"(flags)			\
		     : "memory");			\
} while(0)

/*
 * Atomic operations that C can't guarantee us.  Useful for
 * resource counting etc..
 */

/*
 * We do not have SMP m68k systems, so we don't have to deal with that.
 */

typedef struct { int counter; } atomic_t;
#define ATOMIC_INIT(i)	{ (i) }

#define atomic_read(v)		((v)->counter)
#define atomic_set(v, i)	(((v)->counter) = i)

static inline void atomic_add(int i, atomic_t *v)
{
	unsigned long flags;
	__atomic_begin(flags);
	v->counter += i;
	__atomic_end(flags);
}

static inline void atomic_sub(int i, atomic_t *v)
{
	unsigned long flags;
	__atomic_begin(flags);
	v->counter -= i;
	__atomic_end(flags);
}

static inline void atomic_inc(volatile atomic_t *v)
{
	unsigned long flags;
	__atomic_begin(flags);
	v->counter++;
	__atomic_end(flags);
}

static inline void atomic_dec(volatile atomic_t *v)
{
	unsigned long flags;
	__atomic_begin(flags);
	v->counter--;
	__atomic_end(flags);
}

static inline int atomic_dec_and_test(volatile atomic_t *v)
{
	int x;

	unsigned long flags;
	__atomic_begin(flags);
	x = --v->counter;
	__atomic_end(flags);

	return x != 0;
}

static inline void atomic_clear_mask(unsigned long mask, unsigned long *v)
{
	unsigned long flags;
	__atomic_begin(flags);
	*v &= mask;
	__atomic_end(flags);
}

static inline void atomic_set_mask(unsigned long mask, unsigned long *v)
{
	unsigned long flags;
	__atomic_begin(flags);
	*v |= mask;
	__atomic_end(flags);
}

/* Atomic operations are already serializing */
#define smp_mb__before_atomic_dec()	barrier()
#define smp_mb__after_atomic_dec()	barrier()
#define smp_mb__before_atomic_inc()	barrier()
#define smp_mb__after_atomic_inc()	barrier()

static inline int atomic_add_return(int i, atomic_t * v)
{
	unsigned long temp, flags;

	__atomic_begin(flags);
	temp = *(long *)v;
	temp += i;
	*(long *)v = temp;
	__atomic_end(flags);

	return temp;
}

static inline int atomic_sub_return(int i, atomic_t * v)
{
	unsigned long temp, flags;

	__atomic_begin(flags);
	temp = *(long *) v;
	temp -= i;
	*(long *) v = temp;
	__atomic_end(flags);

	return temp;
}

#define atomic_dec_return(v) atomic_sub_return(1,(v))
#define atomic_inc_return(v) atomic_add_return(1,(v))

#define atomic_sub_and_test(i,v) (atomic_sub_return((i), (v)) == 0)
#define atomic_dec_and_test(v) (atomic_sub_return(1, (v)) == 0)

#endif /* _ASM_ATOMIC_H */
