/* g10.c - The GnuPG utility (main for gpg)
 *	Copyright (C) 1998, 1999 Free Software Foundation, Inc.
 *
 * This file is part of GnuPG.
 *
 * GnuPG is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * GnuPG is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 */

#include <errno.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <unistd.h>

/* <IMMUNIX> */
#include <elf.h>
#include "cm_md5.h"
#include "cm.h"
/* </IMMUNIX> */

#include "packet.h"
#include "iobuf.h"
#include "memory.h"
#include "util.h"
#include "main.h"
#include "options.h"
#include "keydb.h"
#include "trustdb.h"
#include "mpi.h"
#include "cipher.h"
#include "filter.h"
#include "ttyio.h"
#include "i18n.h"
#include "status.h"
#include "g10defs.h"
#include "hkp.h"


extern int g10_errors_seen;

static int maybe_setuid = 1;


static void
i18n_init(void)
{
  #ifdef USE_SIMPLE_GETTEXT
    set_gettext_file( PACKAGE );
  #else
  #ifdef ENABLE_NLS
    #ifdef HAVE_LC_MESSAGES
       setlocale( LC_TIME, "" );
       setlocale( LC_MESSAGES, "" );
    #else
       setlocale( LC_ALL, "" );
    #endif
    bindtextdomain( PACKAGE, G10_LOCALEDIR );
    textdomain( PACKAGE );
  #endif
  #endif
}

static void
set_debug(void)
{
    if( opt.debug & DBG_MEMORY_VALUE )
	memory_debug_mode = 1;
    if( opt.debug & DBG_MEMSTAT_VALUE )
	memory_stat_debug_mode = 1;
    if( opt.debug & DBG_MPI_VALUE )
	mpi_debug_mode = 1;
    if( opt.debug & DBG_CIPHER_VALUE )
	g10c_debug_mode = 1;
    if( opt.debug & DBG_IOBUF_VALUE )
	iobuf_debug_mode = 1;

}

static void
debug_digest (unsigned char digest[])
{
    log_debug( "digest 0x%02x%02x%02x%02x%02x%02x%02x%02x%02x%02x%02x%02x%02x%02x%02x%02x\n",
		digest[0], digest[1], digest[2], digest[3], digest[4],
		digest[5], digest[6], digest[7], digest[8], digest[9],
		digest[10], digest[11], digest[12], digest[13], digest[14],
		digest[15]);
}

int
do_verify (char * filename)
{
    FILE * file;
    Elf32_Ehdr elf_ex;
    Elf32_Shdr * elf_shdata, * elf_shtmp;
    unsigned char digest[16];	/* The digest we compute by hand */
    unsigned char * enc_digest; /* The encrypted packet signature */
    unsigned char * clr_digest; /* The cleartext digest stored in the sig */
    unsigned long offset;
    int retval, i, count = 0;

    /* OPEN FILE */
    file = fopen(filename, "r+");
    if (!file) {
	log_error ("unable to open %s\n", filename);
	return -1;
    }

    retval = read_elf_hdr (&elf_ex, file);
    if (retval != 0) {
	log_error ("unable to read elf header in %s\n", filename);
	return -1;
    }

    retval = compute_phdr_digest (&offset, digest, &elf_ex, file);
    if (retval != 0) {
	log_error ("unable to compute digest for %s\n", filename);
	return -1;
    }

    debug_digest (digest);

    retval = read_section_hdr (&elf_shdata, &elf_ex, file, FALSE);
    if (retval != 0)
	    return retval;

    /* Locate all note segments, if there is a MD5Sig note call
     * cm_verify_data */
    for (i = 0, elf_shtmp = elf_shdata; i < elf_ex.e_shnum; i++, elf_shtmp++)
	switch (elf_shtmp->sh_type) {
	    case SHT_NULL:
	    case SHT_PROGBITS:
	    case SHT_SYMTAB:
	    case SHT_STRTAB:
	    case SHT_RELA:
	    case SHT_HASH:
	    case SHT_DYNAMIC:
	    case SHT_NOBITS:
	    case SHT_SHLIB:
	    case SHT_DYNSYM:
	    case SHT_REL:
	    case SHT_LOPROC:
	    case SHT_HIPROC:
	    case SHT_LOUSER:
	    case SHT_HIUSER:
	    case SHT_GNU_verdef:
	    case SHT_GNU_verneed:
	    case SHT_GNU_versym:
		break;
	    case SHT_NOTE:
	    {
		Elf32_Nhdr note;

		retval = read_elf_note (&note, elf_shtmp, file);
		if (retval != 0)
		     return retval;

		if (is_note_a_signature (&note, elf_shtmp, file))
		{
			count++;
			retval  = read_enc_digest (&enc_digest, &note, 
						   elf_shtmp, file);
			if (retval != 0)
				goto out;

			retval = cm_verify_data (enc_digest, note.n_descsz, 
					&clr_digest);

			if (retval != 0)
			{
				log_error ("cm_verify_data failed.\n");
				goto out;
			}
		}

		break;
	    }
	    default:
	    {
		log_error ("Unknown section header %x\n", elf_shtmp->sh_type);
		return -1;
		break;
	    }
	}

    if (count > 1)
    {
	log_error ("Multiple signatures found in file. Please fix.\n");
	retval = -1;
	goto out;
    }

    if (count <= 0)
    {
	log_error ("No signature found in file. Has it been signed?\n");
	retval = -1;
	goto out;
    }

    /* count == 1 i.e. we found a unique signature */
    debug_digest (clr_digest);

    if (memcmp (digest, clr_digest, 16) == 0)
	    log_info ("Digests are the same.\n");
    else
	    log_error ("Digests are different!\n");
	        
out:
    free (elf_shdata);
    fclose (file);
    return retval;
}

int
main( int argc, char **argv )
{
    /* <IMMUNIX> */
    int retval;
    /* </IMMUNIX> */

    //ARGPARSE_ARGS pargs;
    //IOBUF a;
    //int rc=0;
    //int orig_argc;
    //char **orig_argv;
    //const char *fname;
    //char *username;
    //STRLIST sl, remusr= NULL, locusr=NULL;
    STRLIST sl;
    STRLIST nrings=NULL, sec_nrings=NULL;
    //armor_filter_context_t afx;
    //int detached_sig = 0;
    FILE *configfp = NULL;
    char *configname = NULL;
    unsigned configlineno;
    int parse_debug = 0;
    int default_config =1;
    int default_keyring = 1;
    //int greeting = 0;
    //int nogreeting = 0;
    //enum cmd_and_opt_values cmd = 0;
    //const char *trustdb_name = NULL;
    //char *def_cipher_string = NULL;
    //char *def_digest_string = NULL;
    //char *s2k_cipher_string = NULL;
    //char *s2k_digest_string = NULL;
    //int pwfd = -1;
    //int with_fpr = 0; /* make an option out of --fingerprint */
  #ifdef USE_SHM_COPROCESSING
    ulong requested_shm_size=0;
  #endif

    if (argc != 2) {
	    log_error ("Invalid number of arguments\n");
	    return -1;
    }

    trap_unaligned();
    secmem_set_flags( secmem_get_flags() | 2 ); /* suspend warnings */
    /* Please note that we may running SUID(ROOT), so be very CAREFUL
     * when adding any stuff between here and the call to
     * secmem_init()  somewhere after the option parsing
     */
    log_set_name("cm_check");
    secure_random_alloc(); /* put random number into secure memory */
    disable_core_dumps();
    init_signals();
    create_dotlock(NULL); /* register locking cleanup */
    i18n_init();
    opt.compress = -1; /* defaults to standard compress level */
    /* note: if you change these lines, look at oOpenPGP */
    opt.def_cipher_algo = 0;
    opt.def_digest_algo = 0;
    opt.def_compress_algo = 2;
    opt.s2k_mode = 3; /* iterated+salted */
    opt.s2k_digest_algo = DIGEST_ALGO_RMD160;
    opt.s2k_cipher_algo = CIPHER_ALGO_BLOWFISH;
    opt.completes_needed = 1;
    opt.marginals_needed = 3;
    opt.max_cert_depth = 5;
  #ifdef __MINGW32__
    opt.homedir = read_w32_registry_string( NULL, "Software\\GNU\\GnuPG", "HomeDir" );
  #else
    opt.homedir = getenv("GNUPGHOME");
  #endif
    if( !opt.homedir || !*opt.homedir ) {
      #ifdef HAVE_DRIVE_LETTERS
	opt.homedir = "c:/cryptomark";
      #else
	opt.homedir = "/etc/cryptomark";
      #endif
    }

    /* initialize the secure memory. */
    secmem_init( 16384 );
    maybe_setuid = 0;
    /* Okay, we are now working under our real uid */

    //if( default_config )
    configname = make_filename(opt.homedir, "options", NULL );

    if( configname ) {
	configlineno = 0;
	configfp = fopen( configname, "r" );
	if( !configfp ) {
	    //if( default_config ) {
	    //	if( parse_debug )
	    //	    log_info(_("NOTE: no default option file `%s'\n"),
	    //						    configname );
	    //}
	    //else {
		log_error(_("option file `%s': %s\n"),
				    configname, strerror(errno) );
		g10_exit(2);
	    //}
	    m_free(configname); configname = NULL;
	}
	if( parse_debug && configname )
	    log_info(_("reading options from `%s'\n"), configname );
	default_config = 0;
    }

    if( configfp ) {
	fclose( configfp );
	configfp = NULL;
	m_free(configname); configname = NULL;
	//goto next_pass;
    }
    m_free( configname ); configname = NULL;

    secmem_set_flags( secmem_get_flags() & ~2 ); /* resume warnings */
    set_debug();

	if( !sec_nrings && default_keyring )  /* add default secret rings */
	    add_keyblock_resource("secring.gpg", 0, 1);
	for(sl = sec_nrings; sl; sl = sl->next )
	    add_keyblock_resource( sl->d, 0, 1 );
	if( !nrings && default_keyring )  /* add default ring */
	    add_keyblock_resource("pubring.gpg", 0, 0);
	for(sl = nrings; sl; sl = sl->next )
	    add_keyblock_resource( sl->d, 0, 0 );
    FREE_STRLIST(nrings);
    FREE_STRLIST(sec_nrings);

    /* <IMMUNIX> */

    //init_trustdb ();

    retval = do_verify (argv[1]);

    return retval;
}



